#include <stdio.h>

#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/scrollbar.h>
#include <xview/win_input.h>
#include "list.h"

#include "generic.h"

#include "canvas.h"

list tcanvas_list=NULL;

tcolor tcanvas_Black(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas)
		return(NULL);

	/* returns the tdisplay of the given canvas */
	return((tcolor)tdisplay_Black(c->dpy));
}

tcolor tcanvas_White(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas)
		return(NULL);

	/* returns the tdisplay of the given canvas */
	return((tcolor)tdisplay_White(c->dpy));
}

tdisplay tcanvas_display(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas)
		return(NULL);

	/* returns the tdisplay of the given canvas */
	return(c->dpy);
}

Drawable tcanvas_drawable(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas)
		return(NULL);

	/* returns the window */
	return(c->win);
}

Canvas tcanvas_xview(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas)
		return(NULL);

	/* returns the xview canvas */
	return(c->cvs);
}

int tcanvas_right_side(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(-1);
	if(titem_type(c)!=lt_canvas)
		return(-1);

	/* returns the 1+the right hand coordinate of the canvas */
	if(c->scrolly)
		return(xv_get(c->y, XV_X)+xv_get(c->y, XV_WIDTH));
	else
		return(xv_get(c->cvs, XV_X)+xv_get(c->cvs, XV_WIDTH));
}

int tcanvas_bottom_side(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(-1);
	if(titem_type(c)!=lt_canvas)
		return(-1);

	/* returns the 1+the bottom coordinate of the canvas */
	if(c->scrolly)
		return(xv_get(c->x, XV_Y)+xv_get(c->x, XV_HEIGHT));
	else
		return(xv_get(c->cvs, XV_Y)+xv_get(c->cvs, XV_HEIGHT));
}

int size_to_cvs(cvs, w, h)
/* translates X w and h to canvas coordinates */
ltcanvas cvs;
int *w, *h;
{
	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* adjust for magnification */
    if(cvs->magx>1)
        *w /= cvs->magx;
    else if(cvs->magx<-1)
        *w *= -cvs->magx;
    if(cvs->magy>1)
        *h /= cvs->magy;
    else if(cvs->magy<-1)
        *h *= -cvs->magy;
	
	return(1);
}

int point_to_cvs(cvs, x, y)
/* translates X x and y to canvas coordinates */
ltcanvas cvs;
int *x, *y;
{
	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* transpose */
    *x+=cvs->xpos;
    *y+=cvs->ypos;
    /* adjust for magnification */
    if(cvs->magx>1)
        *x /= cvs->magx;
    else if(cvs->magx<-1)
        *x *= -cvs->magx;
    if(cvs->magy>1)
        *y /= cvs->magy;
    else if(cvs->magy<-1)
        *y *= -cvs->magy;

	return(1);
}

int size_to_norm(cvs, w, h)
/* translates X w and h to un-magnified canvas coordinate system */
ltcanvas cvs;
int *w, *h;
{
	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* adjust for magnification */
    if(cvs->magx>1)
        *w /= cvs->magx;
    else if(cvs->magx<-1)
        *w *= -cvs->magx;
    if(cvs->magy>1)
        *h /= cvs->magy;
    else if(cvs->magy<-1)
        *h *= -cvs->magy;

	return(1);
}

int point_to_norm(cvs, x, y)
/* translates X x and y to un-magnified canvas coordinate system */
ltcanvas cvs;
int *x, *y;
{
	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* adjust for magnification */
    if(cvs->magx>1)
        *x /= cvs->magx;
    else if(cvs->magx<-1)
        *x *= -cvs->magx;
    if(cvs->magy>1)
        *y /= cvs->magy;
    else if(cvs->magy<-1)
        *y *= -cvs->magy;

	return(1);
}

int size_to_x(cvs, w, h)
/* translates cvs w and h to x window coordinates */
ltcanvas cvs;
int *w, *h;
{
    int ow=*w;
    int oh=*h;

	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* adjust for magnification */
    if(cvs->magx>1)
        *w *= cvs->magx;
    else if(cvs->magx<-1)
        *w /= -cvs->magx;
    if(cvs->magy>1)
        *h *= cvs->magy;
    else if(cvs->magy<-1)
        *h /= -cvs->magy;

	return(1);
}

int point_to_x(cvs, x, y)
/* translates cvs x and y to x window coordinates */
ltcanvas cvs;
int *x, *y;
{
	/* check type */
	if(cvs==NULL)
		return(0);
	if(titem_type(cvs)!=lt_canvas)
		return(0);

    /* adjust for magnification */
    if(cvs->magx>1)
        *x *= cvs->magx;
    else if(cvs->magx<-1)
        *x /= -cvs->magx;
    if(cvs->magy>1)
        *y *= cvs->magy;
    else if(cvs->magy<-1)
        *y /= -cvs->magy;
    /* transpose */
    *x-=cvs->xpos;
    *y-=cvs->ypos;

	return(1);
}

int tcanvas_center_x(cvs, w)
/* returns the x coordinate of w if it is to be centered on screen */
ltcanvas cvs;
int w;
{
    int tmpw=cvs->cw/2;
    int tmph;

	/* check type */
	if(cvs==NULL)
		return(-1);
	if(titem_type(cvs)!=lt_canvas)
		return(-1);

    point_to_cvs(cvs, &tmpw, &tmph);
    return(tmpw-w/2);
}

int tcanvas_center_y(cvs, h)
/* returns the y coordinate of h if it is to be centered on screen */
ltcanvas cvs;
int h;
{
    int tmph=cvs->ch/2;
    int tmpw;

	/* check type */
	if(cvs==NULL)
		return(-1);
	if(titem_type(cvs)!=lt_canvas)
		return(-1);

    point_to_cvs(cvs, &tmpw, &tmph);
    return(tmph-h/2);
}

void dorepaintcvs(c, x, y, w, h)
/* do repainting of canvas */
/* x,y,w,h are not magnified or transposed, they're just the redrawable part
    of the x window */
ltcanvas c;
int x, y, w, h;
{
    static char dashes[]={4, 4};
    int px,py;
	int cx,cy,cw,ch;
	XRectangle cliprect;

    px=c->xpos;
    py=c->ypos;

	/* clip to size of canvas */
	if(x<0)
	{
		w+=x;
		x=0;
	}
	if(y<0)
	{
		h+=y;
		y=0;
	}
	if(w>c->cw)
		w=c->cw;
	if(h>c->ch)
		h=c->ch;
	
	/* convert to canvas coordinates, and set clipping area */
	cx=x;
	cy=y;
	ch=h;
	cw=w;
	point_to_cvs(c,&cx,&cy);
	size_to_cvs(c,&cw,&ch);
    tcanvas_set_area(c, cx, cy, cw+1, ch+1);


	/* clear first */
    if(!c->printing)
        XFillRectangle(c->xdpy, c->buf, c->gcerase, x, y, w, h);
	cliprect.x=x;
	cliprect.y=y;
	cliprect.width=w;
	cliprect.height=h;
	XSetClipRectangles(c->xdpy, c->xgc, 0, 0, &cliprect, 1, YXBanded);
	XSetClipRectangles(c->xdpy, c->gcerase, 0, 0, &cliprect, 1, YXBanded);
	XSetClipRectangles(c->xdpy, c->gcpage, 0, 0, &cliprect, 1, YXBanded);


    /* Draw Page boundaries */
    if(c->pages==1 && !c->printing)
    {
        XSetDashes(c->xdpy, c->gcpage, c->ypos+y, dashes, 2);
        px=c->pagew-(c->xpos+x)%c->pagew;
		if(px==c->pagew)
			px=0;
        for(;px<=w; px+=c->pagew)
            XDrawLine(c->xdpy, c->buf, c->gcpage, px+x, y, px+x, y+h);
        XSetDashes(c->xdpy, c->gcpage, c->xpos+x, dashes, 2);
        py=c->pageh-(c->ypos+y)%c->pageh;
		if(py==c->pageh)
			py=0;
        for(;py<=h; py+=c->pageh)
            XDrawLine(c->xdpy, c->buf, c->gcpage, x, py+y, x+w, py+y);
    }

	/* call specific draw routine, if any */
	if(c->do_events!=NULL)
	{
		tevent e;

		/* pass canvas points, not X points */
		e=(tevent)tevent_create_repaint(cx,cy,cw,ch);
		(*c->do_events)(c, e);
		tevent_free(e);
	}
	if(c->back!=NULL && !c->printing)
		XCopyArea(c->xdpy, c->back, c->win, c->gc, x, y, w, h, x, y);
	XSetClipMask(c->xdpy, c->xgc, None);
	XSetClipMask(c->xdpy, c->gcerase, None);
	XSetClipMask(c->xdpy, c->gcpage, None);
}

int tcanvas_pop_buffer(c)
/* puts the area in c->a[xywh] onto the screen */
ltcanvas c;
{
	int x, y, w, h;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	if(c->back==NULL || c->printing)
		return(0);

	x=c->ax;
	y=c->ay;
	w=c->aw;
	h=c->ah;
	point_to_x(c, &x, &y);
	size_to_x(c, &w, &h);
	XCopyArea(c->xdpy, c->back, c->win, c->gc, x, y, w, h, x, y);
}

int tcanvas_magnification(c, magx, magy, px, py)
/* set the magnification levels to magx, magy.  magx=(+/-)2^^n=magy */
/* keeps point px, py from moving on screen */
ltcanvas c;
int magx, magy;
{
    int xlen, ylen, xstart, ystart;
	int i;
	int offx, offy;

	/* check type */
	if(c==NULL)
		return(-1);
	if(titem_type(c)!=lt_canvas)
		return(-1);
	
	/* calculate screen distance from corner */
	if(c->magx>=1)
		offx=px*c->magx-c->xpos;
	else
		offx=px/(-c->magx)-c->xpos;
	if(c->magy>=1)
		offy=py*c->magy-c->ypos;
	else
		offy=py/(-c->magy)-c->ypos;
	/* calculate new x,ypos */
	if(magx>=1)
		c->xpos=px*magx-offx;
	else
		c->xpos=px/(-magx)-offx;
	if(magy>=1)
		c->ypos=py*magy-offy;
	else
		c->ypos=py/(-magy)-offy;
	

    /* first normalize everything */
	if(c->scrollx)
	{
		xlen=xv_get(c->x, SCROLLBAR_OBJECT_LENGTH);
		if(c->magx>1)
		{
			xlen /= c->magx;
			c->pagew /= c->magx;
		}
		else if(c->magx<-1)
		{
			xlen *= -c->magx;
			c->pagew *= -c->magx;
		}
		/* now scale to new values */
		if(magx>1)
		{
			xlen *= magx;
			c->pagew *= magx;
		}
		else if(magx<-1)
		{
			xlen /= -magx;
			c->pagew /= -magx;
		}
	}
	if(c->scrolly)
	{
		ylen=xv_get(c->y, SCROLLBAR_OBJECT_LENGTH);
		if(c->magy>1)
		{
			ylen /= c->magy;
			c->pageh /= c->magy;
		}
		else if(c->magy<-1)
		{
			ylen *= -c->magy;
			c->pageh *= -c->magy;
		}
		/* now scale to new values */
		if(magy>1)
		{
			ylen *= magy;
			c->pageh *= magy;
		}
		else if(magy<-1)
		{
			ylen /= -magy;
			c->pageh /= -magy;
		}
	}
    c->magx=magx;
    c->magy=magy;

	if(magx>1)
		xstart = (c->xpos / magx)/c->cellw;
	else
		xstart = (c->xpos * -magx)/c->cellw;
	if(magy>1)
		ystart = (c->ypos / magy)/c->cellh;
	else
		ystart = (c->ypos * -magy)/c->cellh;
	
	/* now adjust scrollbars */
	if(c->scrollx)
	{
		i=c->pagew/c->cellw;
		if(xstart>xlen-i)
			xstart=xlen-i;
		if(xstart<0)
			xstart=0;
		xv_set(c->x, SCROLLBAR_OBJECT_LENGTH, xlen, NULL);
		xv_set(c->x, SCROLLBAR_VIEW_START, xstart, NULL);
		xv_set(c->x, SCROLLBAR_PAGE_LENGTH, i, NULL);
	}
	if(c->scrolly)
	{
		i=c->pageh/c->cellh;
		if(ystart>ylen-i)
			ystart=ylen-i;
		if(ystart<0)
			ystart=0;
		xv_set(c->y, SCROLLBAR_OBJECT_LENGTH, ylen, NULL);
		xv_set(c->y, SCROLLBAR_VIEW_START, ystart, NULL);
		xv_set(c->y, SCROLLBAR_PAGE_LENGTH, i, NULL);
	}
    /* now repaint */
	if(c->oktodraw)
		dorepaintcvs(c, 0, 0, c->cw, c->ch);
}

int tcanvas_zoom_in(c, px, py)
ltcanvas c;
int px, py;
{
	int mx, my;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(c->magx>=1)
		mx=c->magx*2;
	else if(c->magx<-1)
		mx=c->magx/2;
	else
		mx=2;
	if(c->magy>=1)
		my=c->magy*2;
	else if(c->magy<-1)
		my=c->magy/2;
	else
		my=2;
	tcanvas_magnification(c, mx, my, px, py);
	return(1);
}

int tcanvas_zoom_out(c, px, py)
ltcanvas c;
int px, py;
{
	int mx, my;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(c->magx>1)
		mx=c->magx/2;
	else if(c->magx<=-1)
		mx=c->magx*2;
	else
		mx=-2;
	if(c->magy>1)
		my=c->magy/2;
	else if(c->magy<=-1)
		my=c->magy*2;
	else
		my=-2;
	tcanvas_magnification(c, mx, my, px, py);
	return(1);
}

int tcanvas_zoom_level(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	return(c->magx);
}

int tcanvas_move(mcvs, x,y,right,below)
/* resize an mcvs so if fills the given box */
/* x,y = location in frame in frame to fill,
    right & below are objects to place it right and below of */
ltcanvas mcvs;
int x, y;
titem right, below;
{
	/* check type */
	if(mcvs==NULL)
		return(0);
	if(titem_type(mcvs)!=lt_canvas)
		return(0);

    if(right!=NULL)
    {
		xv_set(mcvs->cvs, XV_X, tright_side(right), NULL);
		if(mcvs->scrollx)
			xv_set(mcvs->x, XV_X, tright_side(right), NULL);
    }
    else
	{
        xv_set(mcvs->cvs, XV_X, x, NULL);
		if(mcvs->scrollx)
			xv_set(mcvs->x, XV_X, x, NULL);
	}
    if(below!=NULL)
    {
		xv_set(mcvs->cvs, XV_Y, tbottom_side(below), NULL);
		if(mcvs->scrolly)
			xv_set(mcvs->y, XV_Y, tbottom_side(below), NULL);
    }
    else
	{
        xv_set(mcvs->cvs, XV_Y, y, NULL);
		if(mcvs->scrolly)
			xv_set(mcvs->y, XV_Y, y, NULL);
	}
	if(mcvs->scrolly)
		xv_set(mcvs->y, WIN_RIGHT_OF, mcvs->cvs, NULL);
	if(mcvs->scrollx)
		xv_set(mcvs->x, WIN_BELOW, mcvs->cvs, NULL);
}

int tcanvas_resize(mcvs, x,y,w,h,right,below)
/* resize an mcvs so if fills the given box */
/* x,y,w,h = box in frame to fill
    right & below are objects to place it right and below of */
ltcanvas mcvs;
int x, y, w, h;
titem right, below;
{
	int nx, ny, nw, nh;
	int i;

	/* check type */
	if(mcvs==NULL)
		return(0);
	if(titem_type(mcvs)!=lt_canvas)
		return(0);

	nx=x;ny=y;nw=w;nh=h;
	if(mcvs->scrolly)
		mcvs->cw=w-xv_get(mcvs->y, XV_WIDTH);
	else
		mcvs->cw=w;
	if(mcvs->scrollx)
		mcvs->ch=h-xv_get(mcvs->x, XV_HEIGHT);
	else
		mcvs->ch=h;
    xv_set(mcvs->cvs, XV_WIDTH, mcvs->cw, NULL);
    xv_set(mcvs->cvs, XV_HEIGHT, mcvs->ch, NULL);
	if(mcvs->scrollx)
		xv_set(mcvs->x, XV_WIDTH, mcvs->cw, NULL);
	if(mcvs->scrolly)
		xv_set(mcvs->y, XV_HEIGHT, mcvs->ch, NULL);
    if(right!=NULL)
    {
		xv_set(mcvs->cvs, XV_X, tright_side(right), NULL);
		if(mcvs->scrollx)
			xv_set(mcvs->x, XV_X, tright_side(right), NULL);
    }
    else
	{
        xv_set(mcvs->cvs, XV_X, x, NULL);
		if(mcvs->scrollx)
			xv_set(mcvs->x, XV_X, x, NULL);
	}
    if(below!=NULL)
    {
		xv_set(mcvs->cvs, XV_Y, tbottom_side(below), NULL);
		if(mcvs->scrolly)
			xv_set(mcvs->y, XV_Y, tbottom_side(below), NULL);
    }
    else
	{
        xv_set(mcvs->cvs, XV_Y, y, NULL);
		if(mcvs->scrolly)
			xv_set(mcvs->y, XV_Y, y, NULL);
	}
	if(mcvs->scrolly)
		xv_set(mcvs->y, WIN_RIGHT_OF, mcvs->cvs, NULL);
	if(mcvs->scrollx)
		xv_set(mcvs->x, WIN_BELOW, mcvs->cvs, NULL);
    mcvs->ch-=2;
    mcvs->cw-=2;
	i=mcvs->cw/mcvs->cellw;
	if(mcvs->scrollx)
	{
		xv_set(mcvs->x, SCROLLBAR_VIEW_LENGTH, i, NULL);
		xv_set(mcvs->x, SCROLLBAR_PAGE_LENGTH, i, NULL);
	}
	i=mcvs->ch/mcvs->cellh;
	if(mcvs->scrolly)
	{
		xv_set(mcvs->y, SCROLLBAR_VIEW_LENGTH, i, NULL);
		xv_set(mcvs->y, SCROLLBAR_PAGE_LENGTH, i, NULL);
	}
	if(mcvs->back!=NULL)
		XFreePixmap(mcvs->xdpy, mcvs->back);
	if(mcvs->buffer)
		mcvs->back=XCreatePixmap(mcvs->xdpy, mcvs->win,
			mcvs->cw, mcvs->ch, mcvs->depth);
	else
		mcvs->back=NULL;
	if(mcvs->back==NULL)
		mcvs->buf=mcvs->win;
	else
		mcvs->buf=mcvs->back;
}

int tcanvas_pages(c, pages)
ltcanvas c;
int pages;
{
	/* check type */
	if(c==NULL)
		return(-1);
	if(titem_type(c)!=lt_canvas)
		return(-1);

	c->pages=pages;
	if(c->oktodraw)
		dorepaintcvs(c, 0, 0, c->cw, c->ch);
}

ltcanvas f_mcvs(pw)
/* return the tcanvas which contains paint window pw */
Canvas pw;
{
    list t=listnode(tcanvas_list);
    ltcanvas current;

    startlist(tcanvas_list);
    while((current=listnext(tcanvas_list))!=NULL)
		if(current->pw==pw)
			break;
    return(current);
}

ltcanvas f_mcvs_scrollbar(s)
/* return the tcanvas which contains scrollbar s */
Scrollbar s;
{
    list t=listnode(tcanvas_list);
    ltcanvas current;

    startlist(tcanvas_list);
    while((current=listnext(tcanvas_list))!=NULL)
        if(current->x==s || current->y==s)
			break;
    return(current);
}

void scrollcvs(c, sc, offset)
/* scroll the canvas (in the direction indicated) by offset cells */
ltcanvas c;
Scrollbar_setting sc;
int offset;
{
    int nx, ny, px, py;

    nx=px=c->xpos;
    ny=py=c->ypos;
    if(sc==SCROLLBAR_VERTICAL)
        ny=offset*c->cellh;
    else
        nx=offset*c->cellw;
	if(ny>c->pageh*c->cvsh-c->ch)
		ny=c->pageh*c->cvsh-c->ch;
	if(ny<0)
		ny=0;
	if(nx>c->pagew*c->cvsw-c->cw)
		nx=c->pagew*c->cvsw-c->cw;
	if(nx<0)
		nx=0;
    if(py==ny && px==nx) return;
    XSetClipMask(c->xdpy, c->gc, None);
    XCopyArea(c->xdpy, c->win, c->win, c->gc,
        0, 0, c->cw, c->ch, px-nx, py-ny);
    c->xpos=nx;
    c->ypos=ny;
    if(sc==SCROLLBAR_VERTICAL)
        if(py<ny)
            dorepaintcvs(c, 0,
                (c->ch-(ny-py)>0)?c->ch-(ny-py):0,
                c->cw,
                (ny-py<c->ch)?ny-py:c->ch); 
        else
            dorepaintcvs(c, 0, 0, c->cw,
                (py-ny<c->ch)?py-ny:c->ch); 
    else
        if(px<nx)
            dorepaintcvs(c,
                (c->cw-(nx-px)>0)?c->cw-(nx-px):0,
                0,
                (nx-px<c->cw)?nx-px:c->cw,
                c->ch);
        else
            dorepaintcvs(c, 0, 0,
                (px-nx<c->cw)?px-nx:c->cw,
                c->ch); 
}

void tcanvas_scroll_down(c)
ltcanvas c;
{
	int viewstart;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	
	if(c->scrolly==0)
		return;
	viewstart=xv_get(c->y, SCROLLBAR_VIEW_START)+1;
	scrollcvs(c, SCROLLBAR_VERTICAL, viewstart);
	xv_set(c->y, SCROLLBAR_VIEW_START, viewstart, NULL);
}

void tcanvas_scroll_up(c)
ltcanvas c;
{
	int viewstart;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	
	if(c->scrolly==0)
		return;
	viewstart=xv_get(c->y, SCROLLBAR_VIEW_START)-1;
	scrollcvs(c, SCROLLBAR_VERTICAL, viewstart);
	xv_set(c->y, SCROLLBAR_VIEW_START, viewstart, NULL);
}

void tcanvas_scroll_to_top(c)
ltcanvas c;
{
	int viewstart;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	
	if(c->scrolly==0)
		return;
	viewstart=0;
	scrollcvs(c, SCROLLBAR_VERTICAL, viewstart);
	xv_set(c->y, SCROLLBAR_VIEW_START, viewstart, NULL);
}

void tcanvas_scroll_right(c)
ltcanvas c;
{
	int viewstart;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	
	if(c->scrollx==0)
		return;
	viewstart=xv_get(c->x, SCROLLBAR_VIEW_START)+1;
	scrollcvs(c, SCROLLBAR_HORIZONTAL, viewstart);
	xv_set(c->x, SCROLLBAR_VIEW_START, viewstart, NULL);
}

void tcanvas_scroll_left(c)
ltcanvas c;
{
	int viewstart;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	
	if(c->scrollx==0)
		return;
	viewstart=xv_get(c->x, SCROLLBAR_VIEW_START)-1;
	scrollcvs(c, SCROLLBAR_HORIZONTAL, viewstart);
	xv_set(c->x, SCROLLBAR_VIEW_START, viewstart, NULL);
}

void tcanvas_set_canvas_width(c, cvsw)
/* set width in pages */
ltcanvas c;
int cvsw;
{
	int i;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	if(cvsw<0)
		return;
	
	c->cvsw=cvsw;
	i=cvsw*c->pagew/c->cellw;
	if(c->scrollx)
		xv_set(c->x, SCROLLBAR_OBJECT_LENGTH, i, NULL);
}

void tcanvas_set_canvas_height(c, cvsh)
/* set height in pages */
ltcanvas c;
int cvsh;
{
	int i;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;
	if(cvsh<0)
		return;
	
	c->cvsh=cvsh;
	i=cvsh*c->pageh/c->cellh;
	if(c->scrolly)
		xv_set(c->y, SCROLLBAR_OBJECT_LENGTH, i, NULL);
}

int tcanvas_point_on_screen(c, x, y)
/* returns 1 if x, y is on screen */
ltcanvas c;
int x, y;
{
	int rx, ry, rw, rh;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return(0);

	rx=c->cw/50;
	ry=c->ch/50;
	rw=c->cw-rx-rx;
	rh=c->ch-ry-ry;
	point_to_x(c, &x, &y);
	return(pt_in_rect(x, y, rx, ry, rw, rh));
}

int tcanvas_point_above_screen(c, y)
/* returns number of cells that y is above the screen, 0 if it's not */
ltcanvas c;
int y;
{
	int x=0;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	point_to_x(c, &x, &y);
	if(y>=0)
		return(0);
	else
	{
		x=(-y)/c->cellh;
		return(x);
	}
}

int tcanvas_point_below_screen(c, y)
/* returns number of cells that y is below the screen, 0 if it's not */
ltcanvas c;
int y;
{
	int x=0;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	point_to_x(c, &x, &y);
	if(y<c->ch)
		return(0);
	else
	{
		x=(y-c->ch)/c->cellh;
		return(x);
	}
}

void tcanvas_move_point_on_screen(c, x, y)
/* scroll the canvas to place x, y on screen */
ltcanvas c;
int x, y;
{
	int rx, ry, rw, rh;
	int mid, offset, newxpos, newypos, newpos;

	if(c==NULL)
		return;
	if(titem_type(c)!=lt_canvas)
		return;

	rx=c->cw/50;
	ry=c->ch/50;
	rw=c->cw-rx-rx;
	rh=c->ch-ry-ry;
	point_to_x(c, &x, &y);

	if(x<rx || x>rx+rw)
	{
		/*must move in x direction*/
		mid=c->cw/2;
		offset=x-mid;
		newpos=c->xpos+offset;
		newpos=newpos/c->cellw;
		scrollcvs(c, SCROLLBAR_HORIZONTAL, newpos);
	}

	if(y<ry || y>ry+rh)
	{
		/*must move in y direction*/
		mid=c->ch/2;
		offset=y-mid;
		newpos=c->ypos+offset;
		newpos=newpos/c->cellh;
		scrollcvs(c, SCROLLBAR_VERTICAL, newpos);
	}

	newxpos=c->xpos/c->cellw;
	newypos=c->ypos/c->cellh;
	if(c->scrollx)
		xv_set(c->x, SCROLLBAR_VIEW_START, newxpos, NULL);
	if(c->scrolly)
		xv_set(c->y, SCROLLBAR_VIEW_START, newypos, NULL);
}

void tcanvas_events(win, e)
/* handle and pass on events in the canvas paint window */
Xv_window win;
Event *e;
{
    ltcanvas cvs;
    int x, y, w, h;

    cvs=f_mcvs(win);
	if(cvs==NULL)
	{
		fprintf(stderr, "Error -- win not from canvas passed\n");
		return;
	}

	/*if(event_action(e)==WIN_GRAPHICS_EXPOSE)*/
	/*{*/
		/*printf("canvas events:WIN_REPAINT\n");*/
		/*return;*/
	/*}*/
	if(event_action(e)==WIN_REPAINT)
	{
		/*printf("canvas events:WIN_REPAINT\n");*/
		return;
	}

	/* call specific event handler */
	if(cvs->do_events!=NULL)
	{
		tevent te;

		/* pass canvas points, not X points */
		x=event_x(e);
		y=event_y(e);
		point_to_cvs(cvs, &x, &y);

		te=(tevent)tevent_create_from_xviewevent(e,x,y,w,h);
		(*cvs->do_events)(cvs, te);
		tevent_free(te);
	}
}

void tcanvas_repaint(cvs, pw, area)
Canvas cvs;
Xv_window pw;
Rectlist *area;
{
    ltcanvas c;

	if(cvs==NULL)
		return;

	c=f_mcvs(pw);
	c->oktodraw=1;
    dorepaintcvs(c,
        area->rl_bound.r_left,
        area->rl_bound.r_top,
        area->rl_bound.r_width,
        area->rl_bound.r_height);
}

int tcanvas_refresh(c, x, y, w, h, all)
ltcanvas c;
int x, y, w, h;
int all;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(all)
		dorepaintcvs(c, 0, 0, c->cw, c->ch);
	else
	{
		point_to_x(c, &x, &y);
		size_to_x(c, &w, &h);
		dorepaintcvs(c, x, y, w, h);
	}
	return(1);
}

void scroll_compute(sc, pos, avail_cable, motion, offset, obj_len)
Scrollbar sc;
int pos;
int avail_cable;
Scroll_motion motion;
unsigned long *offset;
unsigned long *obj_len;
{
    scrollbar_default_compute_scroll_proc(sc,
        pos, avail_cable, motion, offset, obj_len);
    /*ttfprintf(stderr, "motion:%d, pos:%d, ", motion, pos);*/
    scrollcvs(f_mcvs_scrollbar(sc), xv_get(sc, SCROLLBAR_DIRECTION), *offset);
    /*ttfprintf(stderr, "offset:%d\n", *offset);*/
}

tcanvas tcanvas_new2(parent, x,y,w,h,right,below,
	pagew,pageh, cvsw,cvsh, cell_w,cell_h, buffer, scrollx, scrolly, args)
/* make a new canvas */
/* parent=frame, x,y,w,h = box in frame to fill, pagew,pageh = page size in
    pixels, right & below are objects to place it right and below of */
titem parent;
int x,y,w,h;
titem right,below;
int pagew, pageh;
int cvsw, cvsh;
int cell_h, cell_w;
int buffer;
targs args;
int scrollx, scrolly;
{
    ltcanvas tmp;
    XWindowAttributes wa;
    XSetWindowAttributes sa;
    Xv_screen screen;
    int fg,bg;
    unsigned long *colors;
    static char dashes[]={2,2};
	tcolor white, black;

	if(parent==NULL)
		return(NULL);
	if(titem_type(parent)!=lt_frame)
		return(NULL);

    tmp=(ltcanvas)titem_new(parent, lt_canvas, sizeof(ltcanvasd));
    if(tmp==NULL)
    {
        printf("Not enough memory for canvas.\n");
        return(NULL);
    }
	tmp->scrollx=scrollx;
	tmp->scrolly=scrolly;
	tmp->dpy=(tdisplay)tframe_display(parent);
	tmp->xdpy=(Display *)tdisplay_X(tmp->dpy);
	tmp->parent=tframe_xview(parent);
    tmp->realpagew=pagew;
    tmp->realpageh=pageh;
    tmp->pagew=pagew;
    tmp->pageh=pageh;
    tmp->cellh=cell_h;
    tmp->cellw=cell_w;
    tmp->cvsw=cvsw;
    tmp->cvsh=cvsh;
    tmp->cw=w-2;
    tmp->ch=h-2;
    tmp->pages=1;
	tmp->areaset=0;
    tmp->cvs=xv_create(tframe_xview(parent), CANVAS,
        XV_WIDTH, w,
        XV_HEIGHT, h,
        CANVAS_RETAINED, FALSE,
        OPENWIN_AUTO_CLEAR, FALSE,
        CANVAS_AUTO_EXPAND, TRUE,
        CANVAS_AUTO_SHRINK, TRUE,
        CANVAS_WIDTH, w,
        CANVAS_HEIGHT, h,
        CANVAS_REPAINT_PROC, tcanvas_repaint,
		WIN_COLLAPSE_EXPOSURES, TRUE,
        NULL);
    if(tmp->cvs==NULL)
    {
        printf("Couldn't create xview canvas.\n");
        titem_free(tmp);
        return(NULL);
    }
    if(right!=NULL)
		xv_set(tmp->cvs, XV_X, tright_side(right), NULL);
    else
        xv_set(tmp->cvs, XV_X, x, NULL);
    if(below!=NULL)
		xv_set(tmp->cvs, XV_Y, tbottom_side(below), NULL);
    else
        xv_set(tmp->cvs, XV_Y, y, NULL);
	/* now canvas is created and positioned */

	tmp->y=NULL;
	tmp->x=NULL;
	if(scrolly)
	{
		int vl, pl, ol;

		vl=tmp->ch/cell_h;
		pl=pageh/cell_h;
		ol=cvsh*pageh/cell_h;
		tmp->y=xv_create(tframe_xview(parent), SCROLLBAR,
			SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
			SCROLLBAR_PIXELS_PER_UNIT, cell_h,
			SCROLLBAR_VIEW_LENGTH, vl,
			SCROLLBAR_PAGE_LENGTH, pl,
			SCROLLBAR_OBJECT_LENGTH, ol,
			SCROLLBAR_VIEW_START, 0,
			SCROLLBAR_COMPUTE_SCROLL_PROC, scroll_compute,
			WIN_RIGHT_OF, tmp->cvs,
			XV_HEIGHT, xv_get(tmp->cvs, XV_HEIGHT),
			NULL);
		if(tmp->y==NULL)
		{
			printf("Couldn't create xview vertical scrollbar.\n");
			xv_destroy(tmp->cvs);
			titem_free(tmp);
			return(NULL);
		}
		if(below!=NULL)
			xv_set(tmp->y, XV_Y, tbottom_side(below), NULL);
		else
			xv_set(tmp->y, XV_Y, y, NULL);
		xv_set(tmp->y, XV_HEIGHT, h, NULL);
		xv_set(tmp->y, XV_SHOW, TRUE, NULL);
	}
	if(scrollx)
	{
		int vl, pl, ol;

		vl=tmp->cw/cell_w;
		pl=pagew/cell_w;
		ol=cvsw*pagew/cell_w;
		tmp->x=xv_create(tframe_xview(parent), SCROLLBAR,
			SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
			SCROLLBAR_PIXELS_PER_UNIT, cell_w,
			SCROLLBAR_VIEW_LENGTH, vl,
			SCROLLBAR_PAGE_LENGTH, pl,
			SCROLLBAR_OBJECT_LENGTH, ol,
			SCROLLBAR_VIEW_START, 0,
			SCROLLBAR_COMPUTE_SCROLL_PROC, scroll_compute,
			WIN_BELOW, tmp->cvs,
			XV_WIDTH, xv_get(tmp->cvs, XV_WIDTH),
			NULL);
		if(tmp->x==NULL)
		{
			printf("Couldn't create xview horizontal scrollbar.\n");
			xv_destroy(tmp->cvs);
			titem_free(tmp);
			return(NULL);
		}
		if(right!=NULL)
			xv_set(tmp->x, XV_X, tright_side(right), NULL);
		else
			xv_set(tmp->x, XV_X, x, NULL);
		xv_set(tmp->x, XV_WIDTH, w, NULL);
		xv_set(tmp->x, XV_SHOW, TRUE, NULL);
	}

    tmp->pw=canvas_paint_window(tmp->cvs);
    xv_set(tmp->pw,
        WIN_EVENT_PROC, tcanvas_events,
        WIN_CONSUME_EVENTS,
            KBD_DONE, KBD_USE,
            LOC_DRAG, LOC_MOVE, LOC_WINENTER, LOC_WINEXIT,
            WIN_ASCII_EVENTS, WIN_MOUSE_BUTTONS,
            NULL,
        NULL);
    tmp->win=(Window)xv_get(tmp->pw, XV_XID);
    XGetWindowAttributes(tmp->xdpy, tmp->win, &wa);
    tmp->depth=wa.depth;
    screen=(Xv_screen)xv_get(tframe_xview(parent), XV_SCREEN);
    tmp->scr=(int)xv_get(screen, SCREEN_NUMBER);
    tmp->wht=WhitePixel(tmp->xdpy, tmp->scr);
    tmp->blk=BlackPixel(tmp->xdpy, tmp->scr);
    fg=(int)xv_get(tmp->pw, WIN_FOREGROUND_COLOR);
    bg=(int)xv_get(tmp->pw, WIN_BACKGROUND_COLOR);
    colors=(unsigned long *)xv_get(tmp->pw, WIN_X_COLOR_INDICES);
    tmp->fg=colors[fg];
    tmp->bg=colors[bg];
    sa.background_pixel=tmp->bg;
    XChangeWindowAttributes(tmp->xdpy, tmp->win, CWBackPixel, sa);
    tmp->magx=1;
    tmp->magy=1;
    tmp->xpos=0;
    tmp->ypos=0;
    tmp->buffer=buffer;
    if(buffer)
	    tmp->back=XCreatePixmap(tmp->xdpy, tmp->win,
		    tmp->cw, tmp->ch, tmp->depth);
    else
	    tmp->back=NULL;
    if(tmp->back==NULL)
	    tmp->buf=tmp->win;
    else
	    tmp->buf=tmp->back;
    tmp->oktodraw=0;
    tmp->printing=0;

    tmp->gc=XCreateGC(tmp->xdpy, RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->gc);
    XSetForeground(tmp->xdpy, tmp->gc, tmp->fg);
    XSetBackground(tmp->xdpy, tmp->gc, tmp->bg);
    XSetFunction(tmp->xdpy, tmp->gc, GXcopy);
    XSetGraphicsExposures(tmp->xdpy, tmp->gc, True);
    tmp->gcerase=XCreateGC(tmp->xdpy, RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->gcerase);
    XSetForeground(tmp->xdpy, tmp->gcerase, tmp->bg);
    XSetBackground(tmp->xdpy, tmp->gcerase, tmp->bg);
    XSetFunction(tmp->xdpy, tmp->gcerase, GXcopy);
    XSetGraphicsExposures(tmp->xdpy, tmp->gcerase, False);
    tmp->gcpage=XCreateGC(tmp->xdpy, RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->gcpage);
    XSetForeground(tmp->xdpy, tmp->gcpage, tmp->fg);
    XSetBackground(tmp->xdpy, tmp->gcpage, tmp->bg);
    XSetFunction(tmp->xdpy, tmp->gcpage, GXcopy);
    XSetGraphicsExposures(tmp->xdpy, tmp->gcpage, False);
    XSetLineAttributes(tmp->xdpy, tmp->gcpage,
        0, LineOnOffDash, CapButt, JoinBevel);
    tmp->gcdrag=XCreateGC(tmp->xdpy,
		RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->gcdrag);
    XSetForeground(tmp->xdpy, tmp->gcdrag, tmp->fg);
    XSetBackground(tmp->xdpy, tmp->gcdrag, tmp->bg);
    XSetFunction(tmp->xdpy, tmp->gcdrag, GXxor);
    XSetGraphicsExposures(tmp->xdpy, tmp->gcdrag, False);
    XSetLineAttributes(tmp->xdpy, tmp->gcdrag,
        0, LineOnOffDash, CapButt, JoinBevel);
    XSetDashes(tmp->xdpy, tmp->gcdrag, 0, dashes, 2);
    tmp->gcdragsolid=XCreateGC(tmp->xdpy,
		RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->gcdragsolid);
    XSetForeground(tmp->xdpy, tmp->gcdragsolid, tmp->fg);
    XSetBackground(tmp->xdpy, tmp->gcdragsolid, tmp->bg);
    XSetFunction(tmp->xdpy, tmp->gcdragsolid, GXxor);
    XSetGraphicsExposures(tmp->xdpy, tmp->gcdragsolid, False);
    XSetLineAttributes(tmp->xdpy, tmp->gcdragsolid,
        0, LineSolid, CapButt, JoinBevel);

    tmp->xgc=XCreateGC(tmp->xdpy, RootWindow(tmp->xdpy, tmp->scr), 0, NULL);
    tmp->tgc=tcontext_new(parent, args);

    XCopyGC(tmp->xdpy, DefaultGC(tmp->xdpy, DefaultScreen(tmp->xdpy)),
        0xFFFFFFFF, tmp->xgc);

    white=tcolor_white(parent);
    black=tcolor_black(parent);
    XSetForeground(tmp->xdpy, tmp->xgc, tcolor_Xpixel(black));
    tcontext_set_foreground(tmp->tgc, black);
    XSetBackground(tmp->xdpy, tmp->xgc, tcolor_Xpixel(white));
    tcontext_set_background(tmp->tgc, white);

    XSetFunction(tmp->xdpy, tmp->xgc, GXcopy);
    tcontext_set_function(tmp->tgc, tc_copy);

    XSetGraphicsExposures(tmp->xdpy, tmp->xgc, False);

    XSetDashes(tmp->xdpy, tmp->xgc, 0, dashes, 2);
    tcontext_set_dashes(tmp->tgc, 0, 2, 2);

    XSetLineAttributes(tmp->xdpy, tmp->xgc,
        0, LineSolid, CapRound, JoinBevel);
    tcontext_set_linewidth(tmp->tgc, 0);
    tcontext_set_solid(tmp->tgc);
    tcontext_set_capstyle(tmp->tgc, tcap_round);
    tcontext_set_noclipping(tmp->tgc);

    /* now tgc is mirror image of xgc */
	
    tmp->do_events=NULL;

    if(tcanvas_list==NULL)
		tcanvas_list=newlist();
    addnode(tcanvas_list, tmp);

    return(tmp);
}

int tcanvas_free(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);


	if(c->back!=NULL)
		XFreePixmap(c->xdpy, c->back);
	XFreeGC(c->xdpy, c->gc);
	XFreeGC(c->xdpy, c->gcpage);
	XFreeGC(c->xdpy, c->gcdrag);
	XFreeGC(c->xdpy, c->gcdragsolid);
	xv_destroy_safe(c->cvs);
	if(c->scrollx)
		xv_destroy_safe(c->x);
	if(c->scrolly)
		xv_destroy_safe(c->y);
	return(titem_free(c));
}

tcanvas tcanvas_new_no_scrollbars(parent, x,y,w,h,right,below, buffer, args)
titem parent;
int x,y,w,h;
titem right,below;
int buffer;
targs args;
{
	return(tcanvas_new2(parent, x,y,w,h,right,below,
		0,0, 0,0, 0,0, buffer, 0, 0, args));
}

tcanvas tcanvas_new(parent, x,y,w,h,right,below,
	pagew,pageh, cvsw,cvsh, cell_w,cell_h, buffer, args)
titem parent;
int x,y,w,h;
titem right,below;
int pagew, pageh;
int cvsw, cvsh;
int cell_h, cell_w;
int buffer;
targs args;
{
	return(tcanvas_new2(parent, x,y,w,h,right,below,
		pagew,pageh, cvsw,cvsh, cell_w,cell_h, buffer, 1, 1, args));
}

int tcanvas_set_event_procedure(c, do_events)
ltcanvas c;
int (*do_events)();
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	/* setup the event procedure */
	c->do_events=do_events;
	return(1);
}

int tcanvas_set_area(c, x, y, w, h)
/* set clipping area to the given canvas coordinates */
ltcanvas c;
int x,y,w,h;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	c->ax=x;
	c->ay=y;
	c->aw=w;
	c->ah=h;
	c->clipping=1;

	return(1);
}

int tcanvas_in_area(c, x,y,w,h)
/* returns 1 if the given rectangle is in the clipping area */
ltcanvas c;
int x,y,w,h;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(!c->clipping)
		return(1);
	if(x>c->ax+c->aw || x+w<c->ax || y>c->ay+c->ah || y+h<c->ay)
		return(0);
	else
		return(1);
}

int tcanvas_set_clipping(c, context)
/* sets clipping on the context, according to the clipping area for c */
ltcanvas c;
tcontext context;
{
	int x,y,w,h;

	/* check types */
	if(c==NULL || context==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas || titem_type(context)!=lt_context)
		return(0);

	/* convert coordinates */
	point_to_x(c, &x, &y);
	size_to_x(c, &w, &h);

	/* set clipping */
	return(tcontext_set_cliparea(context, x,y,w,h));
}

int tcanvas_start_printing(c, f, filename, type)
ltcanvas c;
FILE *f;
char *filename;
tp_type type;
{
	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	if(type>=tp_none)
		return(0);
	if(f==NULL)
		return(0);
	
	c->printing=1;
	c->printfile=f;
	c->printname=filename;
	c->printtype=type;
	/* actually print to the file */
	pdraw_start(c, f, c->printname, c->realpagew, c->realpageh, type);
	return(1);
}

int tcanvas_stop_printing(c)
ltcanvas c;
{
	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(c->printing)
	{
		/* finish printing to file */
		pdraw_end();
		c->printing=0;
		return(1);
	}
	else
		return(0);
}

int tcanvas_is_printing(c)
ltcanvas c;
{
	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	return(c->printing);
}

FILE *tcanvas_print_file(c)
ltcanvas c;
{
	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(c->printing)
		return(c->printfile);
	else
		return(NULL);
}

int tcanvas_get_pointer(c, x, y, b)
/* returns the pointer location, and true if the button is still down */
ltcanvas c;
int *x, *y;
t_button b;
{
	Window rt, ch;
	int rx, ry;
	unsigned int buttons;

	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	if(!XQueryPointer(c->xdpy, c->win, &rt, &ch, &rx, &ry, x, y, &buttons))
		return(0);

	point_to_cvs(c, x, y);

	switch(b)
	{
		case tb_select:
			if(buttons & Button1Mask)
				return(1);
			else
				return(0);
		case tb_adjust:
			if(buttons & Button2Mask)
				return(1);
			else
				return(0);
		case tb_menu:
			if(buttons & Button3Mask)
				return(1);
			else
				return(0);
		case tb_4:
			if(buttons & Button4Mask)
				return(1);
			else
				return(0);
		case tb_5:
			if(buttons & Button5Mask)
				return(1);
			else
				return(0);
		default:
			return(0);
	}
}

int tcanvas_add_all(c)
ltcanvas c;
{
	int cw, ch;

	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	cw=c->cw;
	ch=c->ch;
	size_to_cvs(c, &cw, &ch);
	return(tcanvas_add_area(c, c->xpos, c->ypos, cw, ch));
}

int tcanvas_add_area(c, x, y, w, h)
ltcanvas c;
int x, y, w, h;
{
	int nx, ny, nw, nh;

	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	normalize_rect(&x, &y, &w, &h);
	if(c->areaset)
	{
		rectunion(&nx,&ny,&nw,&nh,x,y,w,h,c->dx,c->dy,c->dw,c->dh);
		c->dx=nx;
		c->dw=nw;
		c->dh=nh;
		c->dy=ny;
	}
	else
	{
		c->dx=x;
		c->dw=w;
		c->dh=h;
		c->dy=y;
		c->areaset=1;
	}
	return(1);
}

int tcanvas_reset_area(c)
ltcanvas c;
{
	/* check types */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	c->areaset=0;
	return(1);
}

tcanvas_clear_area(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	draw_clear_rect(c, c->dx, c->dy, c->dw+1, c->dh+1);
}

int tcanvas_paint_area(c)
ltcanvas c;
{
	int x, y, w, h;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	/* if area hasn't been set, then don't draw anything */
	if(!c->areaset)
		return(1);

	x=c->dx;
	y=c->dy;
	w=c->dw;
	h=c->dh;
	point_to_x(c, &x, &y);
	size_to_x(c, &w, &h);
	dorepaintcvs(c, x, y, w, h);
	return(1);
}

int tcanvas_clipping_on(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	c->clipping=1;
	return(1);
}

int tcanvas_clipping_off(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	c->clipping=0;
	return(1);
}

int tcanvas_sync(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	XSync(c->xdpy, 1);
}

int tcanvas_grab_pointer(c, e)
ltcanvas c;
tevent e;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	if(e!=NULL)
	{
		if(titem_type(e)!=lt_event)
			return(0);
	}

	if(XGrabPointer(c->xdpy, c->win, True, ButtonPressMask,
			GrabModeAsync, GrabModeAsync, None, None,
			(e==NULL)?CurrentTime:tevent_time(e))==GrabSuccess)
		return(1);
	else
		return(0);
}

int tcanvas_ungrab_pointer(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	XUngrabPointer(c->xdpy, CurrentTime);
}

int tcanvas_add_handle(c, hx, hy, hr)
ltcanvas c;
int hx, hy, hr;
{
	int th;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	size_to_cvs(c, &hr, &th);

	tcanvas_add_area(c, hx-hr-1, hy-hr-1, hr+hr+2, hr+hr+2);
}

int tcanvas_height(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	return(c->ch);
}

int tcanvas_width(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	return(c->cw);
}

int tcanvas_do_timer(pw, w)
Notify_client pw;
int w;
{
	ltcanvas cvs;

	cvs=f_mcvs(pw);
	if(cvs->do_timer!=NULL)
		(*cvs->do_timer)(cvs);
}

int tcanvas_start_timer(c, t, f)
ltcanvas c;
int t;
void (*f)();
{
	static struct itimerval timer;

	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	timer.it_value.tv_usec=t%1000;
	timer.it_value.tv_sec=t/1000;
	timer.it_interval.tv_usec=t%1000;
	timer.it_interval.tv_sec=t/1000;
	c->do_timer=f;
	notify_set_itimer_func(c->pw, tcanvas_do_timer, ITIMER_REAL,
		&timer, NULL);
}

int tcanvas_stop_timer(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);
	
	notify_set_itimer_func(c->pw, NOTIFY_FUNC_NULL, ITIMER_REAL,
		NULL, NULL);
}

Xv_window tcanvas_xview_window(c)
ltcanvas c;
{
	/* check type */
	if(c==NULL)
		return(0);
	if(titem_type(c)!=lt_canvas)
		return(0);

	return(c->pw);
}
