#include <stdio.h>
#include <ctype.h>

#include <xview/xview.h>

#include "generic.h"
#include "memory.h"

typedef struct {
	ltgenericd g;
	tcanvas c;			/* canvas that owns it */
	int x, y, w, h;		/* area on canvas */
	char *text;			/* dynamic string */
	tcontext context;	/* the graphics context for drawing the text */
	tfont font;			/* the font */
	int ssel, esel;		/* start and end of selection */
	int careton;		/* 1 if the caret is on */
	int singleline;		/* 1 if it's to consist of a single line */
	int selected;		/* 1 if it's selected (shows a box around it) */
	int opaque;			/* 1 if it's opaque
							(must erase rectangle before drawing text) */
	unsigned long lasttime;
	tcolor fg, bg;
	} lttextd, *lttext;

int isprintable(c)
int c;
{
	if(isascii(c))
		return(isprint(c));
	else
		return(0);
}

int tskipspace(str, p)
char *str;
int p;
{
	while(p==' ')
		p++;
	return(p);
}

int ttext_fit_line(str, w, f)
char *str;
int w;
tfont f;
{
    int i, j, len, l;

    l=-1;
    i=-1;
    len=0;
    while(len<w)
    {
        i++;
        if(str[i]==' ')
        {
            l=i-1;  /* end of last word that fit */
            i=tskipspace(str, i);
        }
        if(str[i]=='\n' || str[i]=='\r')
        {
            /* everything, including return, will fit */
            return(i+1);
        }
        else if(str[i]=='\0')
        {
            /* whole string will fit, but not null-char */
            return(i);
        }
        else
        {
            len+=tfont_text_width(f, &str[i], 1);
        }
    }
    /* now, l is on end of last word that fit */
    /* now, i is on first non-space character that doesn't fit */
    if(l==-1)
    {
        /* line full of spaces */
        if(i==0)
            i++;
        return(i);
    }
    else
    {
        /* line with words on it */
        /* so include spaces following */
        for(l++;str[l]==' ';l++);
        return(l);
    }
}

int ttext_char_at_pos(t, x, y)
/* returns the index at the given position */
lttext t;
int x, y;
{
    int linenum;
    int len, loc, end;
    int i, w;
    char *thestr;

    thestr=(char *)dstring(t->text);
    linenum=y/tfont_height(t->font);
    len=strlen(thestr);
    loc=0;
    for(i=0;i<linenum;i++)
    {
        loc+=ttext_fit_line(&thestr[loc], t->w, t->font);
        if(loc>=len)
        {
            return(len);
        }
    }
    /* now loc=start of line with point on it */
    end=loc+ttext_fit_line(&thestr[loc], t->w, t->font);
    /* end=#chars on this line */
    for(w=0;loc<end;loc++)
    {
        w+=tfont_text_width(t->font, &thestr[loc], 1);
        if(w>x)
            return(loc);
    }
    return(loc);
}

int ttext_pos_of_char(t, index, x, y)
lttext t;
int index;
int *x, *y;
{
    int loc, last, num_lines, len;
    char *thestr;

    loc=0;
    last=0;
    num_lines=0;
    thestr=(char *)dstring(t->text);
    len=strlen(thestr);
    while(loc<len)
    {
        last=loc;
        loc+=ttext_fit_line(&thestr[loc], t->w, t->font);
        if(loc>index ||
            (loc==index && loc==len))
        {
            *x=tfont_text_width(t->font, &thestr[last], index-last);
            *y=num_lines*tfont_height(t->font)+tfont_ascent(t->font);
            return;
        }
        else if(loc==index)
        {
            *x=0;
            *y=(num_lines+1)*tfont_height(t->font)+tfont_ascent(t->font);
            return;
        }
        num_lines++;
    }
    *x=0;
    *y=tfont_ascent(t->font);
}

ttext_draw_chars(t, s, e)
/* draw chars from s to e */
lttext t;
int s, e;
{
    int loc, last, num_lines, len;
    char *thestr;
    int sx, sl, sp;
    char *str;
    int i;

    loc=0;
    last=0;
    num_lines=0;
    thestr=(char *)dstring(t->text);
    len=strlen(thestr);
    while(loc<e)
    {
        last=loc;
        loc+=ttext_fit_line(&thestr[loc], t->w, t->font);
        if(loc>s && last<e)
        {
            /* draw each line */
            if(last<s)
            {
                sx=tfont_text_width(t->font, &thestr[last], s-last);
                sp=s;
            }
            else
            {
                sx=0;
                sp=last;
            }
            if(loc<e)
                sl=loc-sp;
            else
                sl=e-sp;
            /* now sp=first char, sx=xpos to draw, sl=#chars */
			if(t->opaque)
				draw_text(t->c, t->context,
					t->x+sx,
					t->y+num_lines*tfont_height(t->font)+tfont_ascent(t->font),
					&thestr[sp], sl);
			else
				draw_text_transparent(t->c, t->context,
					t->x+sx,
					t->y+num_lines*tfont_height(t->font)+tfont_ascent(t->font),
					&thestr[sp], sl);
        }
        num_lines++;
    }
}

show_selection(t)
/* show the selection for object t */
lttext t;
{
    int x, y;

    if(t->ssel!=t->esel)
    {
        if(t->careton)
        {
			tcontext_set_foreground(t->context, t->bg);
			tcontext_set_background(t->context, t->fg);
        }
        ttext_draw_chars(t, t->ssel, t->esel);
        if(t->careton)
        {
			tcontext_set_foreground(t->context, t->fg);
			tcontext_set_background(t->context, t->bg);
        }
    }
    else if(t->careton)
    {
        ttext_pos_of_char(t, t->ssel, &x, &y);
        draw_line(t->c, t->context, x+t->x, t->y+y-tfont_ascent(t->font),
								x+t->x, t->y+y+tfont_descent(t->font)-1);
    }
}

int ttext_resize(t, w, h)
/* set the size of the object, changing it's height to accommodate the string */
lttext t;
int *w, *h;
{
    int num_lines, i, loc, len;
    char *thestr;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	if(t->singleline)
	{
		t->w=1+tfont_text_width(t->font, dstring(t->text), dlength(t->text));
		t->h=tfont_height(t->font);
		*w=t->w;
		*h=t->h;
		tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
		return(1);
	}
    loc=0;
    num_lines=0;
    thestr=(char *)dstring(t->text);
    len=strlen(thestr);
	t->w=*w;
	t->h=*h;
    while(loc<len)
    {
        loc+=ttext_fit_line(&thestr[loc], t->w, t->font);
        num_lines++;
    }
    if(num_lines==0)
        num_lines++;
    t->h=num_lines*tfont_height(t->font);
	*w=t->w;
	*h=t->h;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	return(1);
}

int ttext_move_immediately(t, x, y)
lttext t;
int x, y;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	tcanvas_set_area(t->c, t->x-1,t->y-1,t->w+3,t->h+3);
	draw_clear_rect(t->c, t->x, t->y, t->w, t->h);
	t->x=x;
	t->y=y;
	tcanvas_set_area(t->c, t->x-1,t->y-1,t->w+3,t->h+3);
	ttext_repaint(t);
	tcanvas_pop_buffer(t->c);
}

int ttext_move(t, x, y)
lttext t;
int x, y;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	t->x=x;
	t->y=y;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
}

int ttext_repaint(t)
/* redraw the text object */
lttext t;
{
    int i, len;
    char *str;
    int nothing;
	int zoom;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	if(!tcanvas_in_area(t->c, t->x, t->y, t->w, t->h)
			&& !tcanvas_is_printing(t->c))
		return(1);
	str=(char *)dstring(t->text);
    nothing=1;
    len=dlength(t->text);
    for(i=0;i<len;i++)
        if(!(str[i]==' '||str[i]=='\n'||str[i]=='\r'||str[i]=='\0'))
        {
            nothing=0;
            break;
        }

	if(t->selected && !tcanvas_is_printing(t->c))
		draw_rect(t->c, t->context, t->x-1, t->y-1, t->w+2, t->h+2);
	/*zoom=tcanvas_zoom_level(t->c);
	if(zoom>=-1 || tcanvas_is_printing(t->c))
	{
		ttext_draw_chars(t, 0, t->ssel);
		ttext_draw_chars(t, t->esel, dlength(t->text));
		if(t->selected)
			show_selection(t);
	}*/
	ttext_draw_chars(t, 0, t->ssel);
	ttext_draw_chars(t, t->esel, dlength(t->text));
	if(t->selected)
		show_selection(t);

    return(1);
}

ttext_key(t, event)
/* handle key-press event */
lttext t;
Event *event;
{
    int is_key;
    int i;
    char *s;
	int ix,iy,iw,ih,tx,ty,tw,th;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	rectcpy(&ix,&iy,&iw,&ih,t->x,t->y,t->w,t->h);
    switch(event_action(event))
    {
        case ACTION_GO_CHAR_BACKWARD:
            if(t->ssel!=t->esel)
                t->esel=t->ssel;
            else
                if(t->ssel>0)
                    t->ssel=t->esel=t->ssel-1;
            break;
        case ACTION_GO_CHAR_FORWARD:
            if(t->ssel!=t->esel)
                t->ssel=t->esel;
            else
                if(t->ssel<dlength(t->text))
                    t->ssel=t->esel=t->ssel+1;
            break;
        case ACTION_ERASE_CHAR_BACKWARD:
            if(t->ssel!=t->esel)
            {
                del_string(t->text, t->ssel, t->esel-t->ssel);
                t->esel=t->ssel;
            }
            else if(t->ssel>0)
            {
                t->ssel=t->esel=t->ssel-1;
                del_string(t->text, t->ssel, 1);
            }
            break;
        default:
            if(event_is_string(event))
            {
                is_key=1;
                s=event_string(event);
                for(i=0;i<strlen(s);i++)
                    if(!(isprintable(s[i]) ||
						(!t->singleline && (s[i]!='\n' || s[i]!='\r'))))
                        is_key=0;
            }
            else
            {
				is_key=(!t->singleline && ((event_action(event)=='\n') ||
					(event_action(event)=='\r')));
				if(event_is_ascii(event))
					is_key=is_key||isprintable(event_action(event));
            }
            if(is_key)
            {
                if(t->ssel!=t->esel)
                {
                    /* delete selection if there is one */
                    del_string(t->text, t->ssel, t->esel-t->ssel);
                    t->esel=t->ssel;
                }
                if(event_is_string(event))
                {
                    ins_string(t->text, t->ssel, event_string(event));
                    t->ssel=t->esel=t->ssel+strlen(event_string(event));
                }
                else
                {
                    ins_char(t->text, t->ssel, event_action(event));
                    t->ssel=t->esel=t->ssel+1;
                }
            }
            break;
    }
    ttext_resize(t, &t->w, &t->h);
	rectunion(&tx,&ty,&tw,&th,ix,iy,iw,ih,t->x,t->y,t->w,t->h);
	tcanvas_add_area(t->c, tx-1,ty-1,tw+3,th+3);
}

int ttext_get_selection(t, s, e)
/* sets the selection */
lttext t;
int *s, *e;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	*s=t->ssel;
	*e=t->esel;
	return(1);
}

int ttext_set_selection(t, s, e)
/* sets the selection */
lttext t;
int s, e;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	if(s<0)
		s=0;
	if(e<0)
		e=dlength(t->text);
	if(s<e)
	{
		t->ssel=s;
		t->esel=e;
	}
	else
	{
		t->ssel=e;
		t->esel=s;
	}
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	return(1);
}

int ttext_show_caret(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	t->careton=1;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	return(1);
}

int ttext_hide_caret(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	t->careton=0;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	return(1);
}

int ttext_handle_event(t, e)
lttext t;
tevent e;
{
	static int x,y;
	static int dragging=0;
	static int start, loc, nloc;
	static unsigned long newtime;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	tevent_location(e, &x, &y);
	switch(tevent_type(e))
	{
		case te_keydown:
			ttext_key(t, tevent_xview(e));
			break;
		case te_mousedown:
			if(tevent_button(e)==tb_select)
			{
				newtime=tevent_time(e);
				if(newtime-t->lasttime<500)
				{
					t->ssel=0;
					t->esel=dlength(t->text);
				}
				else
				{
					t->lasttime=newtime;
					dragging=1;
					start=t->ssel=t->esel=ttext_char_at_pos(t, x-t->x, y-t->y);
					loc=start;
				}
				tcanvas_add_area(t->c, t->x-1,t->y-1,t->w+5,t->h+3);
			}
			break;
		case te_drag:
			if(dragging)
			{
				nloc=ttext_char_at_pos(t, x-t->x, y-t->y);
				if(nloc!=loc)
				{
					loc=nloc;
					if(loc<start)
					{
						t->ssel=loc;
						t->esel=start;
					}
					else
					{
						t->ssel=start;
						t->esel=loc;
					}
				}
				tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
			}
			break;
		case te_mouseup:
			if(dragging)
			{
				nloc=ttext_char_at_pos(t, x-t->x, y-t->y);
				if(nloc!=loc)
				{
					loc=nloc;
					if(loc<start)
					{
						t->ssel=loc;
						t->esel=start;
					}
					else
					{
						t->ssel=start;
						t->esel=loc;
					}
				}
				tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
				dragging=0;
			}
			break;
		default:
			break;
	}
	return(1);
}

int ttext_change_font(t, f)
lttext t;
tfont f;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	t->font=f;
	tcontext_set_font(t->context, t->font);
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	ttext_resize(t, &t->w, &t->h);
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
	return(1);
}

int ttext_free(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	tcontext_free(t->context);
	free(t);
}

ttext ttext_new(c, x, y, w, h, f, s, fg, bg, args, singleline)
tcanvas c;
int x, y, w, h;
tfont f;
char *s;
tcolor fg, bg;
targs args;
int singleline;
{
	lttext tmp;

	/* check types */
	if(c==NULL || f==NULL || fg==NULL || bg==NULL)
		return(NULL);
	if(titem_type(c)!=lt_canvas ||
			titem_type(f)!=lt_font ||
			titem_type(fg)!=lt_color ||
			titem_type(bg)!=lt_color)
		return(NULL);
	
	tmp=(lttext)titem_new(c, lt_text, sizeof(lttextd));
	if(tmp==NULL)
		return(NULL);

	tmp->c=c;
	tmp->font=f;
	tmp->fg=fg;
	tmp->bg=bg;
	tmp->text=(char *)new_string();
	if(s!=NULL)
		add_string(tmp->text, s);
	tmp->x=x;
	tmp->y=y;
	tmp->singleline=singleline;
	if(singleline)
	{
		h=tfont_height(f);
		w=1+tfont_text_width(f, dstring(tmp->text), dlength(tmp->text));
	}
	tmp->w=w;
	tmp->h=h;
	tmp->careton=1;
	tmp->ssel=tmp->esel=0;
	tmp->selected=0;
	tmp->opaque=1;
	tmp->lasttime=0;
	tmp->context=tcontext_new(c, args);
	tcontext_set_font(tmp->context, f);
	tcontext_set_foreground(tmp->context, fg);
	tcontext_set_background(tmp->context, bg);
	tcontext_set_function(tmp->context, tc_copy);

	return(tmp);	
}

ttext_set_color(t, c)
lttext t;
tcolor c;
{
	/* check types */
	if(t==NULL || c==NULL)
		return(0);
	if(titem_type(t)!=lt_text || titem_type(c)!=lt_color)
		return(0);
	
	t->fg=c;
	tcontext_set_foreground(t->context, c);
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
}

int ttext_insert_string(t, s)
/* insert the string into the text, replacing the selection */
lttext t;
char *s;
{
	int ix,iy,iw,ih,tx,ty,tw,th;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);
	if(s==NULL)
		return(0);


	rectcpy(&ix,&iy,&iw,&ih,t->x,t->y,t->w,t->h);
	if(t->esel!=t->ssel)
	{
		del_string(t->text, t->ssel, t->esel-t->ssel);
		t->esel=t->ssel;
	}
	ins_string(t->text, t->ssel, s);
	t->ssel=t->esel=t->ssel+strlen(s);

    ttext_resize(t, &t->w, &t->h);
	rectunion(&tx,&ty,&tw,&th,ix,iy,iw,ih,t->x,t->y,t->w,t->h);
	tcanvas_add_area(t->c, tx-1,ty-1,tw+3,th+3);
}

int ttext_get_length(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(dlength(t->text));
}

char *ttext_get_string(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return((char *)dstring(t->text));
}

int pt_in_text(t, x, y)
lttext t;
int x, y;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(pt_in_rect(x, y, t->x, t->y, t->w, t->h));
}

int ttext_x(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(t->x);
}

int ttext_y(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(t->y);
}

int ttext_height(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(t->h);
}

int ttext_width(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(t->w);
}

char *ttext_get_selected_string(t)
lttext t;
{
	char *tmp, *sel;
	/* check types */
	if(t==NULL)
		return(NULL);
	if(titem_type(t)!=lt_text)
		return(NULL);

	tmp=(char *)malloc(t->esel-t->ssel+1);
	sel=(char *)dstring(t->text);
	strncpy(tmp, &sel[t->ssel], t->esel-t->ssel);
	tmp[t->esel-t->ssel]='\0';
	return(tmp);
}

int ttext_select(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	t->selected=1;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
}

int ttext_unselect(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	t->selected=0;
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
}

int ttext_selected(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	return(t->selected);
}

int ttext_delete_string(t)
/* deletes the selected string from the text */
lttext t;
{
	int ix,iy,iw,ih,tx,ty,tw,th;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);


	rectcpy(&ix,&iy,&iw,&ih,t->x,t->y,t->w,t->h);
	if(t->esel!=t->ssel)
	{
		del_string(t->text, t->ssel, t->esel-t->ssel);
		t->esel=t->ssel;
	}

    ttext_resize(t, &t->w, &t->h);
	rectunion(&tx,&ty,&tw,&th,ix,iy,iw,ih,t->x,t->y,t->w,t->h);
	tcanvas_add_area(t->c, tx-1,ty-1,tw+3,th+3);
}

ttext ttextcpy(t)
lttext t;
{
	lttext newt;

	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);

	newt=(ttext)titem_new(titem_parent(t), lt_text, sizeof(lttextd));
	newt->c=t->c;
	newt->x=t->x;
	newt->y=t->y;
	newt->w=t->w;
	newt->h=t->h;
	newt->text=(char *)dup_string(t->text);
	newt->context=(tcontext)tcontext_copy(t->context);
	newt->font=t->font;
	newt->ssel=t->ssel;
	newt->esel=t->esel;
	newt->careton=t->careton;
	newt->singleline=t->singleline;
	newt->selected=t->selected;
	newt->fg=t->fg;
	newt->bg=t->bg;
	return((ttext)newt);
}

ttext_add_area(t)
lttext t;
{
	/* check types */
	if(t==NULL)
		return(0);
	if(titem_type(t)!=lt_text)
		return(0);
	
	tcanvas_add_area(t->c, t->x, t->y, t->w, t->h);
}
