from typing import List

from asf_search.constants import PRODUCT_TYPE

NISAR_PRODUCT_TYPES = [
    PRODUCT_TYPE.RRSD, #L0
    PRODUCT_TYPE.SME2, # L3
    PRODUCT_TYPE.GSLC, PRODUCT_TYPE.GCOV, PRODUCT_TYPE.GUNW, PRODUCT_TYPE.GOFF, # L2
    PRODUCT_TYPE.RSLC, PRODUCT_TYPE.RIFG, PRODUCT_TYPE.RUNW, PRODUCT_TYPE.ROFF,  # L1
]

dataset_collections = {
    'ALOS-2': {
        'ALOS2_L1_PSR2': [
            'C1269180392-ASF',
            'C1269180392-ASF',
            'C3315903479-ASF',
        ]
    },
    'NISAR': {
        'NISAR_NEN_RRST_BETA_V1': [
            'C1261815181-ASFDEV',
            'C1261815288-ASF',
            'C2850220296-ASF',
        ],
        'NISAR_NEN_RRST_PROVISIONAL_V1': [
            'C1261832381-ASFDEV',
            'C1261832657-ASF',
            'C2853068083-ASF',
        ],
        'NISAR_NEN_RRST_V1': [
            'C1256533420-ASFDEV',
            'C1257349121-ASF',
            'C2727902012-ASF',
        ],
        'NISAR_L0A_RRST_BETA_V1': [
            'C1261813453-ASFDEV',
            'C1273831241-ASF',
            'C2850223384-ASF',
        ],
        'NISAR_L0A_RRST_PROVISIONAL_V1': [
            'C1261832466-ASFDEV',
            'C1261832658-ASF',
            'C2853086824-ASF',
        ],
        'NISAR_L0A_RRST_V1': [
            'C1256524081-ASFDEV',
            'C1257349120-ASF',
            'C3622203972-ASF',
        ],
        'NISAR_L0B_RRSD_BETA_V1': [
            'C1261815274-ASFDEV',
            'C1273831320-ASF',
            'C3622228339-ASF',
        ],
        'NISAR_L0B_RRSD_PROVISIONAL_V1': [
            'C1261832497-ASFDEV',
            'C1261832659-ASF',
            'C2853089814-ASF',
        ],
        'NISAR_L0B_RRSD_V1': [
            'C1256358262-ASFDEV',
            'C1257349115-ASF',
            'C3622265756-ASF',
        ],
        'NISAR_L1_RSLC_BETA_V1': [
            'C1261813489-ASFDEV',
            'C1273831203-ASF',
            'C2850225585-ASF',
        ],
        'NISAR_L1_RSLC_PROVISIONAL_V1': [
            'C1261832868-ASFDEV',
            'C1261833052-ASF',
            'C2853145197-ASF',
        ],
        'NISAR_L1_RSLC_V1': [
            'C1273095154-ASFDEV',
            'C1273831205-ASF',
            'C3622236985-ASF',
        ],
        'NISAR_L1_RIFG_BETA_V1': [
            'C1261819086-ASFDEV',
            'C1273831200-ASF',
            'C2850234202-ASF',
        ],
        'NISAR_L1_RIFG_PROVISIONAL_V1': [
            'C1261832940-ASFDEV',
            'C1261833063-ASF',
            'C2853147928-ASF',
        ],
        'NISAR_L1_RIFG_V1': [
            'C1256381769-ASFDEV',
            'C1257349108-ASF',
            'C3622229381-ASF',
        ],
        'NISAR_L1_RUNW_BETA_V1': [
            'C1261819098-ASFDEV',
            'C1261819121-ASF',
            'C2850235455-ASF',
        ],
        'NISAR_L1_RUNW_PROVISIONAL_V1': [
            'C1261832990-ASFDEV',
            'C1261833064-ASF',
            'C2853153429-ASF',
        ],
        'NISAR_L1_RUNW_V1': [
            'C1256420738-ASFDEV',
            'C1257349107-ASF',
            'C3622237369-ASF',
        ],
        'NISAR_L1_ROFF_BETA_V1': [
            'C1261819110-ASFDEV',
            'C1261819145-ASF',
            'C2850237619-ASF',
        ],
        'NISAR_L1_ROFF_PROVISIONAL_V1': [
            'C1261832993-ASFDEV',
            'C1261833076-ASF',
            'C2853156054-ASF',
        ],
        'NISAR_L1_ROFF_V1': [
            'C1256411631-ASFDEV',
            'C1257349103-ASF',
            'C3622241997-ASF',
        ],
        'NISAR_L2_GSLC_BETA_V1': [
            'C1261819167-ASFDEV',
            'C1273831198-ASF',
            'C2850259510-ASF',
        ],
        'NISAR_L2_GSLC_PROVISIONAL_V1': [
            'C1261833024-ASFDEV',
            'C1261833127-ASF',
            'C2854332392-ASF',
        ],
        'NISAR_L2_GSLC_V1': [
            'C1256413628-ASFDEV',
            'C1257349102-ASF',
            'C3622244601-ASF',
        ],
        'NISAR_L2_GUNW_BETA_V1': [
            'C1261819168-ASFDEV',
            'C1261819270-ASF',
            'C2850261892-ASF',
        ],
        'NISAR_L2_GUNW_PROVISIONAL_V1': [
            'C1261833025-ASFDEV',
            'C1261846741-ASF',
            'C2854335566-ASF',
        ],
        'NISAR_L2_GUNW_V1': [
            'C1256432264-ASFDEV',
            'C1257349096-ASF',
            'C3622247503-ASF',
        ],
        'NISAR_L2_GCOV_BETA_V1': [
            'C1261819211-ASFDEV',
            'C1273831195-ASF',
            'C3622214170-ASF',
        ],
        'NISAR_L2_GCOV_PROVISIONAL_V1': [
            'C1261833026-ASFDEV',
            'C1261846880-ASF',
            'C2854338529-ASF',
        ],
        'NISAR_L2_GCOV_V1': [
            'C1256477304-ASFDEV',
            'C1257349095-ASF',
            'C2727896018-ASF',
        ],
        'NISAR_L2_GOFF_BETA_V1': [
            'C1261819233-ASFDEV',
            'C1261819281-ASF',
            'C2850263910-ASF',
        ],
        'NISAR_L2_GOFF_PROVISIONAL_V1': [
            'C1261833027-ASFDEV',
            'C1261846994-ASF',
            'C2854341702-ASF',
        ],
        'NISAR_L2_GOFF_V1': [
            'C1256479237-ASFDEV',
            'C1257349094-ASF',
            'C3622250865-ASF',
        ],
        'NISAR_L3_SME2_BETA_V1': [
            'C1261819245-ASFDEV',
            'C1261819282-ASF',
            'C2850265000-ASF',
        ],
        'NISAR_L3_SME2_PROVISIONAL_V1': [
            'C1261833050-ASFDEV',
            'C1261847095-ASF',
            'C2854344945-ASF',
        ],
        'NISAR_L3_SME2_V1': [
            'C1256568692-ASFDEV',
            'C1257349093-ASF',
            'C3622248530-ASF',
        ],
        'NISAR_CUSTOM_PROVISIONAL_V1': [
            'C1262134528-ASFDEV',
            'C1262135006-ASF',
            'C2874824964-ASF',
        ],
        'NISAR_UR_L0B_RRSD': [
            'C1258125097-ASFDEV',
            'C1258836794-ASF',
            'C2887469134-ASF',
        ],
        'NISAR_UR_L1':[
            'C1258290319-ASFDEV',
            'C1273831207-ASF',
            'C2887499907-ASF',
        ],
        'NISAR_UR_L2':[
            'C1258301530-ASFDEV',
            'C1258836800-ASF',
            'C2887502657-ASF',
        ],
        'NISAR_ANC_AUX': [
            'C1261630276-ASFDEV',
            'C1261712879-ASF',
            'C3622222871-ASF',
        ],
        'NISAR_OE': [
            'C1256574227-ASFDEV',
            'C1257349088-ASF',
            'C2727908375-ASF',
        ],
        'NISAR_RP': [
            'C1256579794-ASFDEV',
            'C1257349083-ASF',
            'C2727912431-ASF',
        ],
        'NISAR_LRCLK': [
            'C1256729502-ASFDEV',
            'C1257349067-ASF',
            'C3622253574-ASF',
        ],
        'NISAR_OROST': [
            'C1256578011-ASFDEV',
            'C1257349074-ASF',
            'C2727911748-ASF',
        ],
        'NISAR_TEC': [
            'C1256802323-ASFDEV',
            'C1257349063-ASF',
            'C3622219145-ASF',
        ],
        'NISAR_DC_RADAR': [
            'C1256938783-ASFDEV',
            'C1257349059-ASF',
            'C3622216140-ASF',
        ],
        'NISAR_COP': [
            'C1256535570-ASFDEV',
            'C1257349082-ASF',
            'C2727904608-ASF',
        ],
    },
    'SENTINEL-1': {
        'SENTINEL-1A_SLC': ['C1214470488-ASF', 'C1205428742-ASF', 'C1234413245-ASFDEV'],
        'SENTINEL-1B_SLC': ['C1327985661-ASF', 'C1216244348-ASF', 'C1234413263-ASFDEV'],
        'SENTINEL-1C_SLC': ['C3470873558-ASF', 'C1273088271-ASF',],
        'SENTINEL-1C_DP_GRD_FULL': ['C3486646217-ASF', 'C1273088811-ASF',],
        'SENTINEL-1C_DP_GRD_HIGH': ['C3486566209-ASF', 'C1273088720-ASF',],
        'SENTINEL-1C_DP_GRD_MEDIUM': ['C3486605959-ASF', 'C1273088761-ASF',],
        'SENTINEL-1C_DP_META_GRD_FULL': ['C3486655250-ASF', 'C1273088813-ASF',],
        'SENTINEL-1C_DP_META_GRD_HIGH': ['C3486580531-ASF', 'C1273088738-ASF',],
        'SENTINEL-1C_DP_META_GRD_MEDIUM': ['C3486616327-ASF', 'C1273088779-ASF',],
        'SENTINEL-1C_META_OCN': ['C3484670556-ASF', 'C1273090814-ASF',],
        'SENTINEL-1C_META_RAW': ['C3486527075-ASF', 'C1273090821-ASF',],
        'SENTINEL-1C_META_SLC': ['C3484561061-ASF', 'C1273088273-ASF',],
        'SENTINEL-1C_OCN': ['C3484661603-ASF', 'C1273090807-ASF',],
        'SENTINEL-1C_RAW': ['C3486496642-ASF', 'C1273090818-ASF',],
        'SENTINEL-1C_SP_GRD_FULL': ['C3488389367-ASF', 'C1273088826-ASF',],
        'SENTINEL-1C_SP_GRD_HIGH': ['C3488402208-ASF', 'C1273088882-ASF',],
        'SENTINEL-1C_SP_GRD_MEDIUM': ['C3488414315-ASF', 'C1273090772-ASF',],
        'SENTINEL-1C_SP_META_GRD_FULL': ['C3488394269-ASF', 'C1273088860-ASF',],
        'SENTINEL-1C_SP_META_GRD_HIGH': ['C3488406939-ASF', 'C1273088884-ASF',],
        'SENTINEL-1C_SP_META_GRD_MEDIUM': ['C3488417229-ASF', 'C1273090777-ASF',],
        'SENTINEL-1A_DP_GRD_HIGH': [
            'C1214470533-ASF',
            'C1212201032-ASF',
            'C1234413229-ASFDEV',
        ],
        'SENTINEL-1A_DP_META_GRD_HIGH': [
            'C1214470576-ASF',
            'C1212209226-ASF',
            'C1234413232-ASFDEV',
        ],
        'SENTINEL-1B_DP_GRD_HIGH': [
            'C1327985645-ASF',
            'C1216244589-ASF',
            'C1234413247-ASFDEV',
        ],
        'SENTINEL-1A_META_SLC': [
            'C1214470496-ASF',
            'C1208117434-ASF',
            'C1234413236-ASFDEV',
        ],
        'SENTINEL-1A_META_RAW': [
            'C1214470532-ASF',
            'C1208115009-ASF',
            'C1234413235-ASFDEV',
        ],
        'SENTINEL-1A_OCN': ['C1214472977-ASF', 'C1212212560-ASF', 'C1234413237-ASFDEV'],
        'SENTINEL-1A_DP_META_GRD_MEDIUM': [
            'C1214472336-ASF',
            'C1212212493-ASF',
            'C1234413233-ASFDEV',
        ],
        'SENTINEL-1A_META_OCN': [
            'C1266376001-ASF',
            'C1215704763-ASF',
            'C1234413234-ASFDEV',
        ],
        'SENTINEL-1A_SP_META_GRD_HIGH': [
            'C1214470732-ASF',
            'C1212158326-ASF',
            'C1234413243-ASFDEV',
        ],
        'SENTINEL-1B_DP_GRD_MEDIUM': [
            'C1327985660-ASF',
            'C1216244594-ASF',
            'C1234413248-ASFDEV',
        ],
        'SENTINEL-1B_DP_META_GRD_HIGH': [
            'C1327985741-ASF',
            'C1216244601-ASF',
            'C1234413250-ASFDEV',
        ],
        'SENTINEL-1B_DP_META_GRD_MEDIUM': [
            'C1327985578-ASF',
            'C1216244591-ASF',
            'C1234413251-ASFDEV',
        ],
        'SENTINEL-1B_META_RAW': [
            'C1327985650-ASF',
            'C1216244595-ASF',
            'C1234413253-ASFDEV',
        ],
        'SENTINEL-1B_META_SLC': [
            'C1327985617-ASF',
            'C1216244585-ASF',
            'C1234413254-ASFDEV',
        ],
        'SENTINEL-1B_OCN': ['C1327985579-ASF', 'C1216244593-ASF', 'C1234413255-ASFDEV'],
        'SENTINEL-1B_SP_META_GRD_HIGH': [
            'C1327985619-ASF',
            'C1216244587-ASF',
            'C1234413261-ASFDEV',
        ],
        'SENTINEL-1A_SP_GRD_MEDIUM': [
            'C1214472994-ASF',
            'C1212158318-ASF',
            'C1234413241-ASFDEV',
        ],
        'SENTINEL-1A_SP_META_GRD_MEDIUM': [
            'C1214473170-ASF',
            'C1212233976-ASF',
            'C1234413244-ASFDEV',
        ],
        'SENTINEL-1B_META_OCN': [
            'C1327985646-ASF',
            'C1216244590-ASF',
            'C1234413252-ASFDEV',
        ],
        'SENTINEL-1B_SP_GRD_MEDIUM': [
            'C1327985740-ASF',
            'C1216244600-ASF',
            'C1234413259-ASFDEV',
        ],
        'SENTINEL-1B_SP_META_GRD_MEDIUM': [
            'C1327985739-ASF',
            'C1216244598-ASF',
            'C1234413262-ASFDEV',
        ],
        'SENTINEL-1A_RAW': ['C1214470561-ASF', 'C1205264459-ASF', 'C1234413238-ASFDEV'],
        'SENTINEL-1A_DP_GRD_MEDIUM': [
            'C1214471521-ASF',
            'C1212209035-ASF',
            'C1234413230-ASFDEV',
        ],
        'SENTINEL-1A_SP_GRD_HIGH': [
            'C1214470682-ASF',
            'C1212158327-ASF',
            'C1234413240-ASFDEV',
        ],
        'SENTINEL-1B_RAW': ['C1327985647-ASF', 'C1216244592-ASF', 'C1234413256-ASFDEV'],
        'SENTINEL-1A_DP_GRD_FULL': [
            'C1214471197-ASF',
            'C1212200781-ASF',
            'C1234413228-ASFDEV',
        ],
        'SENTINEL-1A_DP_META_GRD_FULL': [
            'C1214471960-ASF',
            'C1212209075-ASF',
            'C1234413231-ASFDEV',
        ],
        'SENTINEL-1A_SP_GRD_FULL': ['C1214472978-ASF', 'C1234413239-ASFDEV'],
        'SENTINEL-1A_SP_META_GRD_FULL': ['C1214473165-ASF', 'C1234413242-ASFDEV'],
        'SENTINEL-1B_DP_GRD_FULL': [
            'C1327985697-ASF',
            'C1216244597-ASF',
            'C1234413246-ASFDEV',
        ],
        'SENTINEL-1B_DP_META_GRD_FULL': [
            'C1327985651-ASF',
            'C1216244596-ASF',
            'C1234413249-ASFDEV',
        ],
        'SENTINEL-1B_SP_GRD_FULL': [
            'C1327985644-ASF',
            'C1216244588-ASF',
            'C1234413257-ASFDEV',
        ],
        'SENTINEL-1B_SP_GRD_HIGH': [
            'C1327985571-ASF',
            'C1216244586-ASF',
            'C1234413258-ASFDEV',
        ],
        'SENTINEL-1B_SP_META_GRD_FULL': [
            'C1327985674-ASF',
            'C1216244599-ASF',
            'C1234413260-ASFDEV',
        ],
        'S1_Bursts': ['C1244552887-ASFDEV'],
        'SENTINEL-1_BURSTS_DEV10': ['C1257175154-ASFDEV'],
        'Sentinel-1_Burst_Map': ['C1244598379-ASFDEV'],
        'Various Browse Images': ['C1240784657-ASFDEV'],
    },
    'OPERA-S1': {
        'OPERA_L2_CSLC-S1_V1': ['C2777443834-ASF', 'C1259976861-ASF'],
        'OPERA_L2_RTC-S1_V1': ['C2777436413-ASF', 'C1259974840-ASF'],
        'OPERA_L2_CSLC-S1-STATIC_PROVISIONAL_V0': ['C1258354200-ASF'],
        'OPERA_L2_CSLC-S1-STATIC_V1': ['C1259982010-ASF', 'C2795135668-ASF'],
        'OPERA_L2_CSLC-S1_PROVISIONAL_V0': ['C1257995185-ASF'],
        'OPERA_L2_RTC-S1-STATIC_PROVISIONAL_V0': ['C1258354201-ASF'],
        'OPERA_L2_RTC-S1-STATIC_V1': ['C1259981910-ASF', 'C2795135174-ASF'],
        'OPERA_L2_RTC-S1_PROVISIONAL_V0': ['C1257995186-ASF'],
        'OPERA_L3_DISP-S1_V1': ['C3294057315-ASF', 'C1271830354-ASF'],
        'OPERA_L4_TROPO-ZENITH_V1': ['C3717139408-ASF'],
    },
    'OPERA-S1-CALVAL': {
        'OPERA_L2_CSLC-S1_CALVAL_V1': ['C1260721945-ASF', 'C2803501758-ASF'],
        'OPERA_L2_RTC-S1_CALVAL_V1': ['C1260721853-ASF', 'C2803501097-ASF'],
    },
    'SLC-BURST': {'SENTINEL-1_BURSTS': ['C2709161906-ASF', 'C1257024016-ASF']},
    'ALOS PALSAR': {
        'ALOS_PSR_RTC_HIGH': ['C1206487504-ASF', 'C1207181535-ASF'],
        'ALOS_PSR_L1.5': ['C1206485940-ASF', 'C1205261223-ASF'],
        'ALOS_PSR_RTC_LOW': ['C1206487217-ASF', 'C1208013295-ASF'],
        'ALOS_PSR_KMZ': ['C1206156901-ASF', 'C1207019609-ASF'],
        'ALOS_PSR_L1.0': ['C1206485320-ASF'],
        'ALOS_PSR_L1.1': ['C1206485527-ASF', 'C1207710476-ASF', 'C1239611505-ASFDEV'],
        'ALOS_PSR_L2.2': ['C2011599335-ASF', 'C1239927797-ASF', 'C1238733834-ASFDEV'],
        'ALOS_PALSAR_INSAR_METADATA': ['C1229740239-ASF'],
    },
    'ALOS AVNIR-2': {
        'ALOS_AVNIR_OBS_ORI': [
            'C1808440897-ASF',
            'C1233629671-ASF',
            'C1234413224-ASFDEV',
        ],
        'ALOS_AVNIR_OBS_ORI_BROWSE': ['C1234712303-ASF'],
    },
    'SIR-C': {
        'STS-59_BROWSE_GRD': [
            'C1661710578-ASF',
            'C1226557819-ASF',
            'C1234413264-ASFDEV',
        ],
        'STS-59_BROWSE_SLC': [
            'C1661710581-ASF',
            'C1226557809-ASF',
            'C1234413265-ASFDEV',
        ],
        'STS-59_GRD': ['C1661710583-ASF', 'C1226557808-ASF', 'C1234413266-ASFDEV'],
        'STS-59_META_GRD': ['C1661710586-ASF', 'C1226557810-ASF', 'C1234413267-ASFDEV'],
        'STS-59_META_SLC': ['C1661710588-ASF', 'C1226557811-ASF', 'C1234413268-ASFDEV'],
        'STS-59_SLC': ['C1661710590-ASF', 'C1226557812-ASF', 'C1234413269-ASFDEV'],
        'STS-68_BROWSE_GRD': [
            'C1661710593-ASF',
            'C1226557813-ASF',
            'C1234413270-ASFDEV',
        ],
        'STS-68_BROWSE_SLC': [
            'C1661710596-ASF',
            'C1226557814-ASF',
            'C1234413271-ASFDEV',
        ],
        'STS-68_GRD': ['C1661710597-ASF', 'C1226557815-ASF', 'C1234413272-ASFDEV'],
        'STS-68_META_GRD': ['C1661710600-ASF', 'C1226557816-ASF', 'C1234413273-ASFDEV'],
        'STS-68_META_SLC': ['C1661710603-ASF', 'C1226557817-ASF', 'C1234413274-ASFDEV'],
        'STS-68_SLC': ['C1661710604-ASF', 'C1226557818-ASF', 'C1234413275-ASFDEV'],
    },
    'ARIA S1 GUNW': {
        'SENTINEL-1_INTERFEROGRAMS': ['C1595422627-ASF', 'C1225776654-ASF'],
        'SENTINEL-1_INTERFEROGRAMS_AMPLITUDE': ['C1596065640-ASF', 'C1225776655-ASF'],
        'SENTINEL-1_INTERFEROGRAMS_COHERENCE': ['C1596065639-ASF', 'C1225776657-ASF'],
        'SENTINEL-1_INTERFEROGRAMS_CONNECTED_COMPONENTS': [
            'C1596065641-ASF',
            'C1225776658-ASF',
        ],
        'SENTINEL-1_INTERFEROGRAMS_UNWRAPPED_PHASE': [
            'C1595765183-ASF',
            'C1225776659-ASF',
        ],
        'ARIA_S1_GUNW': ['C2859376221-ASF', 'C1261881077-ASF'],
    },
    'SMAP': {
        'SPL1A_RO_METADATA_003': ['C1243122884-ASF', 'C1233103964-ASF'],
        'SPL1A_RO_QA_003': ['C1243124139-ASF', 'C1216074923-ASF'],
        'SPL1A_001': ['C1214473171-ASF', 'C1212243761-ASF'],
        'SPL1A_002': ['C1243149604-ASF', 'C1213091807-ASF'],
        'SPL1A_METADATA_001': ['C1214473426-ASF', 'C1212243437-ASF'],
        'SPL1A_METADATA_002': ['C1243119801-ASF', 'C1213096699-ASF'],
        'SPL1A_QA_001': ['C1214473839-ASF', 'C1212249653-ASF'],
        'SPL1A_QA_002': ['C1243133204-ASF', 'C1213101573-ASF'],
        'SPL1A_RO_001': ['C1243197402-ASF'],
        'SPL1A_RO_002': ['C1243215430-ASF', 'C1213136240-ASF'],
        'SPL1A_RO_003': ['C1243124754-ASF', 'C1216074755-ASF'],
        'SPL1A_RO_METADATA_001': ['C1243141638-ASF', 'C1213136752-ASF'],
        'SPL1A_RO_METADATA_002': ['C1243162394-ASF', 'C1213136799-ASF'],
        'SPL1A_RO_QA_001': ['C1243168733-ASF', 'C1213136709-ASF'],
        'SPL1A_RO_QA_002': ['C1243168866-ASF', 'C1213136844-ASF'],
        'SPL1B_SO_LoRes_001': ['C1214473308-ASF', 'C1212249811-ASF'],
        'SPL1B_SO_LoRes_002': ['C1243253631-ASF', 'C1213125007-ASF'],
        'SPL1B_SO_LoRes_003': ['C1243133445-ASF', 'C1216074919-ASF'],
        'SPL1B_SO_LoRes_METADATA_001': ['C1214473550-ASF', 'C1212196951-ASF'],
        'SPL1B_SO_LoRes_METADATA_002': ['C1243197502-ASF', 'C1213115690-ASF'],
        'SPL1B_SO_LoRes_METADATA_003': ['C1243126328-ASF', 'C1216074758-ASF'],
        'SPL1B_SO_LoRes_QA_001': ['C1214474243-ASF', 'C1212243666-ASF'],
        'SPL1B_SO_LoRes_QA_002': ['C1243216659-ASF', 'C1213115896-ASF'],
        'SPL1B_SO_LoRes_QA_003': ['C1243129847-ASF', 'C1216074761-ASF'],
        'SPL1C_S0_HiRes_001': ['C1214473367-ASF', 'C1212250364-ASF'],
        'SPL1C_S0_HiRes_002': ['C1243268956-ASF', 'C1213134622-ASF'],
        'SPL1C_S0_HiRes_003': ['C1243144528-ASF', 'C1216074770-ASF'],
        'SPL1C_S0_HiRes_METADATA_001': ['C1214473624-ASF', 'C1212246173-ASF'],
        'SPL1C_S0_HiRes_METADATA_002': ['C1243228612-ASF', 'C1213125156-ASF'],
        'SPL1C_S0_HiRes_METADATA_003': ['C1243136142-ASF', 'C1216074764-ASF'],
        'SPL1C_S0_HiRes_QA_001': ['C1214474435-ASF', 'C1212249773-ASF'],
        'SPL1C_S0_HiRes_QA_002': ['C1243255360-ASF', 'C1213134486-ASF'],
        'SPL1C_S0_HiRes_QA_003': ['C1243140611-ASF', 'C1233101609-ASF'],
        'SPL1A_003': ['C1216074922-ASF'],
        'SPL1A_METADATA_003': ['C1216074750-ASF'],
        'SPL1A_QA_003': ['C1216074751-ASF'],
    },
    'UAVSAR': {
        'UAVSAR_POL_META': ['C1214353986-ASF', 'C1210487703-ASF'],
        'UAVSAR_INSAR_META': ['C1214336717-ASF', 'C1212030772-ASF'],
        'UAVSAR_INSAR_INT': ['C1214336045-ASF', 'C1212001698-ASF'],
        'UAVSAR_INSAR_AMP': ['C1214335430-ASF', 'C1206116665-ASF'],
        'UAVSAR_INSAR_AMP_GRD': ['C1214335471-ASF', 'C1206132445-ASF'],
        'UAVSAR_INSAR_DEM': ['C1214335903-ASF', 'C1211962154-ASF'],
        'UAVSAR_INSAR_INT_GRD': ['C1214336154-ASF', 'C1212005594-ASF'],
        'UAVSAR_INSAR_KMZ': ['C1214336554-ASF', 'C1212019993-ASF'],
        'UAVSAR_POL_DEM': ['C1214353593-ASF', 'C1207638502-ASF'],
        'UAVSAR_POL_INC': ['C1214353754-ASF', 'C1210025872-ASF'],
        'UAVSAR_POL_KMZ': ['C1214353859-ASF', 'C1210485039-ASF'],
        'UAVSAR_POL_ML_CMPLX_GRD': ['C1214337770-ASF', 'C1207188317-ASF'],
        'UAVSAR_POL_ML_CMPLX_GRD_3X3': ['C1214354144-ASF', 'C1210546638-ASF'],
        'UAVSAR_POL_ML_CMPLX_GRD_5X5': ['C1214354235-ASF', 'C1206122195-ASF'],
        'UAVSAR_POL_ML_CMPLX_SLANT': ['C1214343609-ASF', 'C1209970710-ASF'],
        'UAVSAR_POL_PAULI': ['C1214354031-ASF', 'C1207038647-ASF'],
        'UAVSAR_POL_SLOPE': ['C1214408428-ASF', 'C1210599503-ASF'],
        'UAVSAR_POL_STOKES': ['C1214419355-ASF', 'C1210599673-ASF'],
    },
    'RADARSAT-1': {
        'RSAT-1_L0': ['C1206897141-ASF'],
        'RSAT-1_L1': ['C1206936391-ASF', 'C1205181982-ASF'],
        'RSAT-1_POLAR_YEAR_ANTARCTICA_L1': ['C1215670813-ASF'],
        'RSAT-1_POLAR_YEAR_GREENLAND_L0': ['C1215709884-ASF'],
        'RSAT-1_POLAR_YEAR_GREENLAND_L1': ['C1215709880-ASF'],
        'RSAT-1_POLAR_YEAR_KAMCHATKA_L1': ['C1215714443-ASF'],
        'RSAT-1_POLAR_YEAR_SEA_ICE_MIN_MAX_L1': ['C1215775284-ASF'],
        'RSAT-1_POLAR_YEAR_TOOLIK_L1': ['C1215614037-ASF'],
    },
    'ERS': {
        'ERS-1_L0': ['C1210197768-ASF', 'C1205261222-ASF'],
        'ERS-1_L1': ['C1211627521-ASF', 'C1205302527-ASF'],
        'ERS-2_L0': ['C1208794942-ASF', 'C1207143701-ASF'],
        'ERS-2_L1': ['C1209373626-ASF', 'C1207144966-ASF'],
    },
    'JERS-1': {
        'JERS-1_L0': ['C1208662092-ASF', 'C1207175327-ASF'],
        'JERS-1_L1': ['C1207933168-ASF', 'C1207177736-ASF'],
    },
    'AIRSAR': {
        'AIRSAR_POL_3FP': ['C1213921661-ASF', 'C1205256880-ASF'],
        'AIRSAR_INT_JPG': ['C1213921626-ASF', 'C1000000306-ASF'],
        'AIRSAR_POL_SYN_3FP': ['C1213928843-ASF', 'C1208713702-ASF'],
        'AIRSAR_TOP_C-DEM_STOKES': ['C1213927035-ASF', 'C1208707768-ASF'],
        'AIRSAR_TOP_DEM': ['C179001730-ASF', 'C1208655639-ASF'],
        'AIRSAR_TOP_DEM_C': ['C1213925022-ASF', 'C1208680681-ASF'],
        'AIRSAR_TOP_DEM_L': ['C1213926419-ASF', 'C1208691361-ASF'],
        'AIRSAR_TOP_DEM_P': ['C1213926777-ASF', 'C1208703384-ASF'],
        'AIRSAR_TOP_L-STOKES': ['C1213927939-ASF'],
        'AIRSAR_TOP_P-STOKES': ['C1213928209-ASF'],
        'AIRSAR_INT': ['C1208652494-ASF'],
    },
    'SEASAT': {
        'SEASAT_L1_SAR': ['C3576379529-ASF', 'C1271768606-ASF'],
    },
}

collections_per_platform = {
    'SENTINEL-1A': [
        'C2803501758-ASF',
        'C2803501097-ASF',
        'C1214470488-ASF',
        'C1214470533-ASF',
        'C1214470576-ASF',
        'C1595422627-ASF',
        'C2859376221-ASF',
        'C1261881077-ASF',
        'C1214470496-ASF',
        'C1214470532-ASF',
        'C1214472977-ASF',
        'C1214472336-ASF',
        'C1266376001-ASF',
        'C1214472994-ASF',
        'C1214470732-ASF',
        'C1214473170-ASF',
        'C1214470561-ASF',
        'C1214471521-ASF',
        'C1214470682-ASF',
        'C2777443834-ASF',
        'C2777436413-ASF',
        'C1214471197-ASF',
        'C1214471960-ASF',
        'C1214472978-ASF',
        'C1214473165-ASF',
        'C2709161906-ASF',
        'C1596065640-ASF',
        'C1596065639-ASF',
        'C1596065641-ASF',
        'C1595765183-ASF',
        'C2450786986-ASF',
        'C1205428742-ASF',
        'C1212201032-ASF',
        'C1212212560-ASF',
        'C1205264459-ASF',
        'C1208117434-ASF',
        'C1212209035-ASF',
        'C1212209226-ASF',
        'C1208115009-ASF',
        'C1212158327-ASF',
        'C1215704763-ASF',
        'C1225776654-ASF',
        'C1212158318-ASF',
        'C1212212493-ASF',
        'C1212158326-ASF',
        'C1212233976-ASF',
        'C1260726384-ASF',
        'C1258354200-ASF',
        'C1259982010-ASF',
        'C2795135668-ASF',
        'C1260721945-ASF',
        'C1257995185-ASF',
        'C1259976861-ASF',
        'C1258354201-ASF',
        'C1259981910-ASF',
        'C2795135174-ASF',
        'C1260721853-ASF',
        'C1257995186-ASF',
        'C1259974840-ASF',
        'C1212200781-ASF',
        'C1212209075-ASF',
        'C1257024016-ASF',
        'C1225776655-ASF',
        'C1225776657-ASF',
        'C1225776658-ASF',
        'C1225776659-ASF',
        'C1245953394-ASF',
        'C1234413245-ASFDEV',
        'C1234413229-ASFDEV',
        'C1234413237-ASFDEV',
        'C1234413238-ASFDEV',
        'C1234413236-ASFDEV',
        'C1234413230-ASFDEV',
        'C1234413232-ASFDEV',
        'C1234413235-ASFDEV',
        'C1234413240-ASFDEV',
        'C1234413234-ASFDEV',
        'C1234413241-ASFDEV',
        'C1234413233-ASFDEV',
        'C1234413243-ASFDEV',
        'C1234413244-ASFDEV',
        'C1244552887-ASFDEV',
        'C1234413228-ASFDEV',
        'C1234413231-ASFDEV',
        'C1234413239-ASFDEV',
        'C1234413242-ASFDEV',
        'C1257175154-ASFDEV',
        'C1244598379-ASFDEV',
        'C1240784657-ASFDEV',
    ],
    'SENTINEL-1B': [
        'C2803501758-ASF',
        'C2803501097-ASF',
        'C1327985661-ASF',
        'C1327985645-ASF',
        'C1595422627-ASF',
        'C1327985617-ASF',
        'C1327985660-ASF',
        'C1327985741-ASF',
        'C1327985578-ASF',
        'C1327985646-ASF',
        'C1327985650-ASF',
        'C1327985579-ASF',
        'C1327985740-ASF',
        'C1327985619-ASF',
        'C1327985739-ASF',
        'C1327985647-ASF',
        'C2777443834-ASF',
        'C2777436413-ASF',
        'C1327985697-ASF',
        'C1327985651-ASF',
        'C1327985644-ASF',
        'C1327985571-ASF',
        'C1327985674-ASF',
        'C2709161906-ASF',
        'C1596065640-ASF',
        'C1596065639-ASF',
        'C1596065641-ASF',
        'C1595765183-ASF',
        'C2450786986-ASF',
        'C1216244348-ASF',
        'C1216244589-ASF',
        'C1216244594-ASF',
        'C1216244593-ASF',
        'C1216244585-ASF',
        'C1216244592-ASF',
        'C1216244595-ASF',
        'C1225776654-ASF',
        'C1216244590-ASF',
        'C1216244601-ASF',
        'C1216244600-ASF',
        'C1216244591-ASF',
        'C1216244587-ASF',
        'C1216244598-ASF',
        'C1216244586-ASF',
        'C1260726384-ASF',
        'C1258354200-ASF',
        'C1259982010-ASF',
        'C2795135668-ASF',
        'C1260721945-ASF',
        'C1257995185-ASF',
        'C1259976861-ASF',
        'C1258354201-ASF',
        'C1259981910-ASF',
        'C2795135174-ASF',
        'C1260721853-ASF',
        'C1257995186-ASF',
        'C1259974840-ASF',
        'C1216244597-ASF',
        'C1216244596-ASF',
        'C1216244588-ASF',
        'C1216244599-ASF',
        'C1257024016-ASF',
        'C1225776655-ASF',
        'C1225776657-ASF',
        'C1225776658-ASF',
        'C1225776659-ASF',
        'C1245953394-ASF',
        'C1234413263-ASFDEV',
        'C1234413247-ASFDEV',
        'C1234413248-ASFDEV',
        'C1234413255-ASFDEV',
        'C1234413254-ASFDEV',
        'C1234413256-ASFDEV',
        'C1234413253-ASFDEV',
        'C1234413252-ASFDEV',
        'C1234413250-ASFDEV',
        'C1234413259-ASFDEV',
        'C1234413251-ASFDEV',
        'C1234413261-ASFDEV',
        'C1234413262-ASFDEV',
        'C1234413258-ASFDEV',
        'C1244552887-ASFDEV',
        'C1234413246-ASFDEV',
        'C1234413249-ASFDEV',
        'C1234413257-ASFDEV',
        'C1234413260-ASFDEV',
        'C1257175154-ASFDEV',
        'C1244598379-ASFDEV',
    ],
    'SENTINEL-1C': [
        'C3486646217-ASF',
        'C3486566209-ASF',
        'C3486605959-ASF',
        'C3486655250-ASF',
        'C3486580531-ASF',
        'C3486616327-ASF',
        'C3484670556-ASF',
        'C3486527075-ASF',
        'C3484561061-ASF',
        'C3484661603-ASF',
        'C3486496642-ASF',
        'C3470873558-ASF',
        'C3488389367-ASF',
        'C3488402208-ASF',
        'C3488414315-ASF',
        'C3488394269-ASF',
        'C3488406939-ASF',
        'C3488417229-ASF',
        'C2709161906-ASF',
        "C1273090818-ASF", # UAT
        "C1273090821-ASF",
        "C1273088271-ASF",
        "C1273088273-ASF",
        "C1273090807-ASF",
        "C1273090814-ASF",
        "C1273088720-ASF",
        "C1273088738-ASF",
        "C1273088761-ASF",
        "C1273088779-ASF",
        "C1273088811-ASF",
        "C1273088813-ASF",
        "C1273088882-ASF",
        "C1273088884-ASF",
        "C1273090772-ASF",
        "C1273090777-ASF",
        "C1273088826-ASF",
        "C1273088860-ASF",
    ],
    'STS-59': [
        'C1661710578-ASF',
        'C1661710581-ASF',
        'C1661710583-ASF',
        'C1661710586-ASF',
        'C1661710588-ASF',
        'C1661710590-ASF',
        'C1226557819-ASF',
        'C1226557809-ASF',
        'C1226557808-ASF',
        'C1226557810-ASF',
        'C1226557811-ASF',
        'C1226557812-ASF',
        'C1234413264-ASFDEV',
        'C1234413265-ASFDEV',
        'C1234413266-ASFDEV',
        'C1234413267-ASFDEV',
        'C1234413268-ASFDEV',
        'C1234413269-ASFDEV',
    ],
    'STS-68': [
        'C1661710593-ASF',
        'C1661710596-ASF',
        'C1661710597-ASF',
        'C1661710600-ASF',
        'C1661710603-ASF',
        'C1661710604-ASF',
        'C1226557813-ASF',
        'C1226557814-ASF',
        'C1226557815-ASF',
        'C1226557816-ASF',
        'C1226557817-ASF',
        'C1226557818-ASF',
        'C1234413270-ASFDEV',
        'C1234413271-ASFDEV',
        'C1234413272-ASFDEV',
        'C1234413273-ASFDEV',
        'C1234413274-ASFDEV',
        'C1234413275-ASFDEV',
    ],
    'ALOS': [
        'C1206487504-ASF',
        'C1206487217-ASF',
        'C1206485940-ASF',
        'C1206156901-ASF',
        'C1206485320-ASF',
        'C1206485527-ASF',
        'C1808440897-ASF',
        'C2011599335-ASF',
        'C1207181535-ASF',
        'C1207710476-ASF',
        'C1234712303-ASF',
        'C1239927797-ASF',
        'C1205261223-ASF',
        'C1233629671-ASF',
        'C1208013295-ASF',
        'C1207019609-ASF',
        'C1229740239-ASF',
        'C1239611505-ASFDEV',
        'C1238733834-ASFDEV',
        'C1234413224-ASFDEV',
    ],
    "ALOS-2": [
        'C1269180392-ASF',
        'C1269180392-ASF',
        "C3315903479-ASF",
    ],
    'ERS-1': [
        'C1210197768-ASF',
        'C1211627521-ASF',
        'C1205261222-ASF',
        'C1205302527-ASF',
    ],
    'ERS-2': [
        'C1208794942-ASF',
        'C1209373626-ASF',
        'C1207143701-ASF',
        'C1207144966-ASF',
    ],
    'JERS-1': [
        'C1208662092-ASF',
        'C1207933168-ASF',
        'C1207175327-ASF',
        'C1207177736-ASF',
    ],
    'RADARSAT-1': [
        'C1206897141-ASF',
        'C1206936391-ASF',
        'C1205181982-ASF',
        'C1215670813-ASF',
        'C1215709884-ASF',
        'C1215709880-ASF',
        'C1215714443-ASF',
        'C1215775284-ASF',
        'C1215614037-ASF',
    ],
    'DC-8': [
        'C1213921661-ASF',
        'C1213921626-ASF',
        'C1213928843-ASF',
        'C1213927035-ASF',
        'C179001730-ASF',
        'C1213925022-ASF',
        'C1213926419-ASF',
        'C1213926777-ASF',
        'C1213927939-ASF',
        'C1213928209-ASF',
        'C1205256880-ASF',
        'C1208652494-ASF',
        'C1000000306-ASF',
        'C1208713702-ASF',
        'C1208707768-ASF',
        'C1208655639-ASF',
        'C1208680681-ASF',
        'C1208691361-ASF',
        'C1208703384-ASF',
    ],
    'SEASAT 1': [
        'C3576379529-ASF', # prod
        'C1271768606-ASF', # uat
    ],
    'SMAP': [
        'C1243122884-ASF',
        'C1243124139-ASF',
        'C1214473171-ASF',
        'C1243149604-ASF',
        'C1214473426-ASF',
        'C1243119801-ASF',
        'C1214473839-ASF',
        'C1243133204-ASF',
        'C1243197402-ASF',
        'C1243215430-ASF',
        'C1243124754-ASF',
        'C1243141638-ASF',
        'C1243162394-ASF',
        'C1243168733-ASF',
        'C1243168866-ASF',
        'C1214473308-ASF',
        'C1243253631-ASF',
        'C1243133445-ASF',
        'C1214473550-ASF',
        'C1243197502-ASF',
        'C1243126328-ASF',
        'C1214474243-ASF',
        'C1243216659-ASF',
        'C1243129847-ASF',
        'C1214473367-ASF',
        'C1243268956-ASF',
        'C1243144528-ASF',
        'C1214473624-ASF',
        'C1243228612-ASF',
        'C1243136142-ASF',
        'C1214474435-ASF',
        'C1243255360-ASF',
        'C1243140611-ASF',
        'C1233103964-ASF',
        'C1216074923-ASF',
        'C1212243761-ASF',
        'C1213091807-ASF',
        'C1216074922-ASF',
        'C1212243437-ASF',
        'C1213096699-ASF',
        'C1216074750-ASF',
        'C1212249653-ASF',
        'C1213101573-ASF',
        'C1216074751-ASF',
        'C1213136240-ASF',
        'C1216074755-ASF',
        'C1213136752-ASF',
        'C1213136799-ASF',
        'C1213136709-ASF',
        'C1213136844-ASF',
        'C1212249811-ASF',
        'C1213125007-ASF',
        'C1216074919-ASF',
        'C1212196951-ASF',
        'C1213115690-ASF',
        'C1216074758-ASF',
        'C1212243666-ASF',
        'C1213115896-ASF',
        'C1216074761-ASF',
        'C1212250364-ASF',
        'C1213134622-ASF',
        'C1216074770-ASF',
        'C1212246173-ASF',
        'C1213125156-ASF',
        'C1216074764-ASF',
        'C1212249773-ASF',
        'C1213134486-ASF',
        'C1233101609-ASF',
    ],
    'G-III': [
        'C1214353986-ASF',
        'C1214336045-ASF',
        'C1214336717-ASF',
        'C1214335430-ASF',
        'C1214335471-ASF',
        'C1214335903-ASF',
        'C1214336154-ASF',
        'C1214336554-ASF',
        'C1214353593-ASF',
        'C1214353754-ASF',
        'C1214353859-ASF',
        'C1214337770-ASF',
        'C1214354144-ASF',
        'C1214354235-ASF',
        'C1214343609-ASF',
        'C1214354031-ASF',
        'C1214408428-ASF',
        'C1214419355-ASF',
        'C1210487703-ASF',
        'C1212030772-ASF',
        'C1206116665-ASF',
        'C1206132445-ASF',
        'C1211962154-ASF',
        'C1212001698-ASF',
        'C1212005594-ASF',
        'C1212019993-ASF',
        'C1207638502-ASF',
        'C1210025872-ASF',
        'C1210485039-ASF',
        'C1207188317-ASF',
        'C1210546638-ASF',
        'C1206122195-ASF',
        'C1209970710-ASF',
        'C1207038647-ASF',
        'C1210599503-ASF',
        'C1210599673-ASF',
    ],
    'NISAR': [
        # UAT ASFDEV
        'C1261815181-ASFDEV',
        'C1261832381-ASFDEV',
        'C1256533420-ASFDEV',
        'C1261813453-ASFDEV',
        'C1261832466-ASFDEV',
        'C1256524081-ASFDEV',
        'C1261815274-ASFDEV',
        'C1261832497-ASFDEV',
        'C1256358262-ASFDEV',
        'C1261813489-ASFDEV',
        'C1261832868-ASFDEV',
        'C1273095154-ASFDEV',
        'C1261819086-ASFDEV',
        'C1261832940-ASFDEV',
        'C1256381769-ASFDEV',
        'C1261819098-ASFDEV',
        'C1261832990-ASFDEV',
        'C1256420738-ASFDEV',
        'C1261819110-ASFDEV',
        'C1261832993-ASFDEV',
        'C1256411631-ASFDEV',
        'C1261819167-ASFDEV',
        'C1261833024-ASFDEV',
        'C1256413628-ASFDEV',
        'C1261819168-ASFDEV',
        'C1261833025-ASFDEV',
        'C1256432264-ASFDEV',
        'C1261819211-ASFDEV',
        'C1261833026-ASFDEV',
        'C1256477304-ASFDEV',
        'C1261819233-ASFDEV',
        'C1261833027-ASFDEV',
        'C1256479237-ASFDEV',
        'C1261819245-ASFDEV',
        'C1261833050-ASFDEV',
        'C1256568692-ASFDEV',
        'C1262134528-ASFDEV',
        'C1258125097-ASFDEV',
        'C1258290319-ASFDEV',
        'C1258301530-ASFDEV',
        'C1261630276-ASFDEV',
        'C1256574227-ASFDEV',
        'C1256579794-ASFDEV',
        'C1256729502-ASFDEV',
        'C1256535568-ASFDEV',
        'C1256578011-ASFDEV',
        'C1256802323-ASFDEV',
        'C1256938783-ASFDEV',
        'C1256535570-ASFDEV',
        # UAT
        'C1261815288-ASF',
        'C1261832657-ASF',
        'C1257349121-ASF',
        'C1273831241-ASF',
        'C1261832658-ASF',
        'C1257349120-ASF',
        'C1273831320-ASF',
        'C1261832659-ASF',
        'C1257349115-ASF',
        'C1273831203-ASF',
        'C1261833052-ASF',
        'C1273831205-ASF',
        'C1273831200-ASF',
        'C1261833063-ASF',
        'C1257349108-ASF',
        'C1261819121-ASF',
        'C1261833064-ASF',
        'C1257349107-ASF',
        'C1261819145-ASF',
        'C1261833076-ASF',
        'C1257349103-ASF',
        'C1273831198-ASF',
        'C1261833127-ASF',
        'C1257349102-ASF',
        'C1261819270-ASF',
        'C1261846741-ASF',
        'C1257349096-ASF',
        'C1273831195-ASF',
        'C1261846880-ASF',
        'C1257349095-ASF',
        'C1261819281-ASF',
        'C1261846994-ASF',
        'C1257349094-ASF',
        'C1261819282-ASF',
        'C1261847095-ASF',
        'C1257349093-ASF',
        'C1262135006-ASF',
        'C1258836794-ASF',
        'C1273831207-ASF',
        'C1258836800-ASF',
        'C1261712879-ASF',
        'C1257349088-ASF',
        'C1257349083-ASF',
        'C1257349067-ASF',
        'C1257349089-ASF',
        'C1257349074-ASF',
        'C1257349063-ASF',
        'C1257349059-ASF',
        'C1257349082-ASF',
        # PROD
        'C2850220296-ASF',
        'C2853068083-ASF',
        'C2727902012-ASF',
        'C2850223384-ASF',
        'C2853086824-ASF',
        'C3622203972-ASF',
        'C3622228339-ASF',
        'C2853089814-ASF',
        'C3622265756-ASF',
        'C2850225585-ASF',
        'C2853145197-ASF',
        'C3622236985-ASF',
        'C2850234202-ASF',
        'C2853147928-ASF',
        'C3622229381-ASF',
        'C2850235455-ASF',
        'C2853153429-ASF',
        'C3622237369-ASF',
        'C2850237619-ASF',
        'C2853156054-ASF',
        'C3622241997-ASF',
        'C2850259510-ASF',
        'C2854332392-ASF',
        'C3622244601-ASF',
        'C2850261892-ASF',
        'C2854335566-ASF',
        'C3622247503-ASF',
        'C3622214170-ASF',
        'C2854338529-ASF',
        'C2727896018-ASF',
        'C2850263910-ASF',
        'C2854341702-ASF',
        'C3622250865-ASF',
        'C2850265000-ASF',
        'C2854344945-ASF',
        'C3622248530-ASF',
        'C2874824964-ASF',
        'C2887469134-ASF',
        'C2887499907-ASF',
        'C2887502657-ASF',
        'C3622222871-ASF',
        'C2727908375-ASF',
        'C2727912431-ASF',
        'C3622253574-ASF',
        'C3622233495-ASF',
        'C2727911748-ASF',
        'C3622219145-ASF',
        'C3622216140-ASF',
        'C2727904608-ASF',
    ],
}


collections_by_processing_level = {
    'SLC': [
        'C1214470488-ASF',
        'C1205428742-ASF',
        'C1234413245-ASFDEV',
        'C1327985661-ASF',
        'C1216244348-ASF',
        'C1234413263-ASFDEV',
        'C1661710588-ASF',
        'C1661710590-ASF',
        'C1226557811-ASF',
        'C1226557812-ASF',
        'C1661710603-ASF',
        'C1661710604-ASF',
        'C1226557817-ASF',
        'C1226557818-ASF',
        'C3470873558-ASF',  # S1C
        'C1273088271-ASF'
    ],
    'GRD_HD': [
        'C1214470533-ASF',
        'C1212201032-ASF',
        'C1234413229-ASFDEV',
        'C1327985645-ASF',
        'C1216244589-ASF',
        'C3486566209-ASF',  # S1C
        'C1273088720-ASF',
    ],
    'METADATA_GRD_HD': [
        'C1214470576-ASF',
        'C1212209226-ASF',
        'C1234413232-ASFDEV',
        'C1327985741-ASF',
        'C1216244601-ASF',
        'C3486580531-ASF',  # S1C
        'C1273088738-ASF',
    ],
    'GUNW_STD': [
        'C1595422627-ASF',
        'C1225776654-ASF',
        'C1595422627-ASF',
        'C1225776654-ASF',
    ],
    'METADATA_SLC': [
        'C1214470496-ASF',
        'C1208117434-ASF',
        'C1234413236-ASFDEV',
        'C1327985617-ASF',
        'C1216244585-ASF',
        'C1234413254-ASFDEV',
        'C3484561061-ASF',  # S1C
        'C1273088273-ASF',
    ],
    'METADATA_RAW': [
        'C1214470532-ASF',
        'C1208115009-ASF',
        'C1234413235-ASFDEV',
        'C1327985650-ASF',
        'C1216244595-ASF',
        'C3486527075-ASF',  # S1C
        'C1273090821-ASF',
    ],
    'OCN': [
        'C1214472977-ASF',
        'C1212212560-ASF',
        'C1234413237-ASFDEV',
        'C1327985579-ASF',
        'C1216244593-ASF',
        'C1234413255-ASFDEV',
        'C3484661603-ASF',  # S1C
        'C1273090807-ASF',
    ],
    'METADATA_GRD_MD': [
        'C1214472336-ASF',
        'C1212212493-ASF',
        'C1234413233-ASFDEV',
        'C1327985578-ASF',
        'C1216244591-ASF',
        'C3486616327-ASF',  # S1C
        'C1273088779-ASF',
    ],
    'METADATA_OCN': [
        'C1266376001-ASF',
        'C1215704763-ASF',
        'C1234413234-ASFDEV',
        'C1327985646-ASF',
        'C1216244590-ASF',
        'C1234413252-ASFDEV',
        'C3484670556-ASF',  # S1C
        'C1273090814-ASF',
    ],
    'GRD_MS': [
        'C1214472994-ASF',
        'C1212158318-ASF',
        'C1327985740-ASF',
        'C1216244600-ASF',
        'C3488414315-ASF',  # S1C
        'C1273090772-ASF',
    ],
    'METADATA_GRD_HS': [
        'C1214470732-ASF',
        'C1212158326-ASF',
        'C1234413243-ASFDEV',
        'C1327985619-ASF',
        'C1216244587-ASF',
        'C3488406939-ASF',  # S1C
        'C1273088884-ASF',
    ],
    'METADATA_GRD_MS': [
        'C1214473170-ASF',
        'C1212233976-ASF',
        'C1327985739-ASF',
        'C1216244598-ASF',
        'C3488417229-ASF',  # S1C
        'C1273090777-ASF',
    ],
    'RAW': [
        'C1214470561-ASF',
        'C1205264459-ASF',
        'C1234413238-ASFDEV',
        'C1327985647-ASF',
        'C1216244592-ASF',
        'C1234413256-ASFDEV',
        'C3486496642-ASF',  # S1C
        'C1273090818-ASF',
    ],
    'GRD_MD': [
        'C1214471521-ASF',
        'C1212209035-ASF',
        'C1234413230-ASFDEV',
        'C1327985660-ASF',
        'C1216244594-ASF',
        'C3486605959-ASF',  # S1C
        'C1273088761-ASF',
    ],
    'GRD_HS': [
        'C1214470682-ASF',
        'C1212158327-ASF',
        'C1234413240-ASFDEV',
        'C1327985571-ASF',
        'C1216244586-ASF',
        'C3488402208-ASF',  # S1C
        'C1273088882-ASF',
    ],
    'CSLC': [
        'C2777443834-ASF',
        'C1260721945-ASF',
        'C2803501758-ASF',
        'C1259976861-ASF',
    ],
    'RTC': [
        'C2777436413-ASF',
        'C1260721853-ASF',
        'C2803501097-ASF',
        'C1259974840-ASF',
    ],
    'DISP-S1': [
        'C3294057315-ASF',
        'C1271830354-ASF'
    ],
    'TROPO-ZENITH': [
        'C3717139408-ASF',
    ],
    'GRD_FD': [
        'C1214471197-ASF',
        'C1212200781-ASF',
        'C3486646217-ASF',  # S1C
        'C1273088811-ASF',
    ],
    'METADATA_GRD_FD': [
        'C1214471960-ASF',
        'C1212209075-ASF',
        'C3486655250-ASF',  # S1C
        'C1273088813-ASF',
    ],
    'BURST': [
        'C2709161906-ASF',
        'C1257024016-ASF',
        'C1257175154-ASFDEV',
    ],
    'GUNW_AMP': [
        'C1596065640-ASF',
        'C1225776655-ASF',
        'C1596065640-ASF',
        'C1225776655-ASF',
    ],
    'GUNW_COH': [
        'C1596065639-ASF',
        'C1225776657-ASF',
        'C1596065639-ASF',
        'C1225776657-ASF',
    ],
    'GUNW_CON': [
        'C1596065641-ASF',
        'C1225776658-ASF',
        'C1596065641-ASF',
        'C1225776658-ASF',
    ],
    'GUNW_UNW': [
        'C1595765183-ASF',
        'C1225776659-ASF',
        'C1595765183-ASF',
        'C1225776659-ASF',
    ],
    'CSLC-STATIC': ['C1259982010-ASF', 'C2795135668-ASF'],
    'RTC-STATIC': ['C1259981910-ASF', 'C2795135174-ASF'],
    'GRD': [
        'C1661710583-ASF',
        'C1661710586-ASF',
        'C1226557808-ASF',
        'C1226557810-ASF',
        'C1661710597-ASF',
        'C1661710600-ASF',
        'C1226557815-ASF',
        'C1226557816-ASF',
    ],
    'RTC_HI_RES': ['C1206487504-ASF', 'C1207181535-ASF'],
    'RTC_LOW_RES': ['C1206487217-ASF', 'C1208013295-ASF'],
    'L1.5': ['C1206485940-ASF', 'C1205261223-ASF'],
    'KMZ': [
        'C1206156901-ASF',
        'C1207019609-ASF',
        'C1214336554-ASF',
        'C1214353859-ASF',
        'C1212019993-ASF',
        'C1210485039-ASF',
    ],
    'L1.0': ['C1206485320-ASF'],
    'L1.1': ['C1206485527-ASF', 'C1207710476-ASF', 'C1239611505-ASFDEV'],
    'L2.2': ['C2011599335-ASF', 'C1239927797-ASF', 'C1238733834-ASFDEV'],
    'L0': [
        'C1210197768-ASF',
        'C1205261222-ASF',
        'C1208794942-ASF',
        'C1207143701-ASF',
        'C1207933168-ASF',
        'C1207175327-ASF',
        'C1206897141-ASF',
    ],
    'L1': [
        'C1211627521-ASF',
        'C1205302527-ASF',
        'C1209373626-ASF',
        'C1207144966-ASF',
        'C1208662092-ASF',
        'C1207177736-ASF',
        'C1206936391-ASF',
        'C1205181982-ASF',
    ],
    '3FP': ['C1213921661-ASF', 'C1213928843-ASF', 'C1205256880-ASF', 'C1208713702-ASF'],
    'JPG': ['C1213921626-ASF', 'C1000000306-ASF'],
    'CSTOKES': ['C1213927035-ASF', 'C1208707768-ASF'],
    'DEM': ['C179001730-ASF', 'C1208655639-ASF'],
    'CTIF': ['C1213925022-ASF', 'C1208680681-ASF'],
    'LTIF': ['C1213926419-ASF', 'C1208691361-ASF'],
    'PTIF': ['C1213926777-ASF', 'C1208703384-ASF'],
    'LSTOKES': ['C1213927939-ASF'],
    'PSTOKES': ['C1213928209-ASF'],
    'ATI': ['C1208652494-ASF'],
    'L1A_Radar_RO_ISO_XML': [
        'C1243122884-ASF',
        'C1243141638-ASF',
        'C1243162394-ASF',
        'C1233103964-ASF',
        'C1213136752-ASF',
        'C1213136799-ASF',
    ],
    'L1A_Radar_RO_QA': [
        'C1243124139-ASF',
        'C1243168733-ASF',
        'C1243168866-ASF',
        'C1216074923-ASF',
        'C1213136709-ASF',
        'C1213136844-ASF',
    ],
    'L1A_Radar_HDF5': [
        'C1214473171-ASF',
        'C1243149604-ASF',
        'C1212243761-ASF',
        'C1213091807-ASF',
    ],
    'L1A_Radar_ISO_XML': [
        'C1214473426-ASF',
        'C1243119801-ASF',
        'C1212243437-ASF',
        'C1213096699-ASF',
    ],
    'L1A_Radar_QA': [
        'C1214473839-ASF',
        'C1243133204-ASF',
        'C1212249653-ASF',
        'C1213101573-ASF',
    ],
    'L1A_Radar_RO_HDF5': [
        'C1243197402-ASF',
        'C1243215430-ASF',
        'C1243124754-ASF',
        'C1213136240-ASF',
        'C1216074755-ASF',
    ],
    'L1B_S0_LoRes_HDF5': [
        'C1214473308-ASF',
        'C1243253631-ASF',
        'C1243133445-ASF',
        'C1212249811-ASF',
        'C1213125007-ASF',
        'C1216074919-ASF',
    ],
    'L1B_S0_LoRes_ISO_XML': [
        'C1214473550-ASF',
        'C1243197502-ASF',
        'C1243126328-ASF',
        'C1212196951-ASF',
        'C1213115690-ASF',
        'C1216074758-ASF',
    ],
    'L1B_S0_LoRes_QA': [
        'C1214474243-ASF',
        'C1243216659-ASF',
        'C1243129847-ASF',
        'C1212243666-ASF',
        'C1213115896-ASF',
        'C1216074761-ASF',
    ],
    'L1C_S0_HiRes_HDF5': [
        'C1214473367-ASF',
        'C1243268956-ASF',
        'C1243144528-ASF',
        'C1212250364-ASF',
        'C1213134622-ASF',
        'C1216074770-ASF',
    ],
    'L1C_S0_HiRes_ISO_XML': [
        'C1214473624-ASF',
        'C1243228612-ASF',
        'C1243136142-ASF',
        'C1212246173-ASF',
        'C1213125156-ASF',
        'C1216074764-ASF',
    ],
    'L1C_S0_HiRes_QA': [
        'C1214474435-ASF',
        'C1243255360-ASF',
        'C1243140611-ASF',
        'C1212249773-ASF',
        'C1213134486-ASF',
        'C1233101609-ASF',
    ],
    'METADATA': [
        'C1214353986-ASF',
        'C1214336717-ASF',
        'C1210487703-ASF',
        'C1212030772-ASF',
    ],
    'INTERFEROMETRY': ['C1214336045-ASF', 'C1212001698-ASF'],
    'AMPLITUDE': ['C1214335430-ASF', 'C1206116665-ASF'],
    'AMPLITUDE_GRD': ['C1214335471-ASF', 'C1206132445-ASF'],
    'DEM_TIFF': [
        'C1214335903-ASF',
        'C1214353593-ASF',
        'C1211962154-ASF',
        'C1207638502-ASF',
    ],
    'INTERFEROMETRY_GRD': ['C1214336154-ASF', 'C1212005594-ASF'],
    'INC': ['C1214353754-ASF', 'C1210025872-ASF'],
    'PROJECTED': ['C1214337770-ASF', 'C1207188317-ASF'],
    'PROJECTED_ML3X3': ['C1214354144-ASF', 'C1210546638-ASF'],
    'PROJECTED_ML5X5': ['C1214354235-ASF', 'C1206122195-ASF'],
    'COMPLEX': ['C1214343609-ASF', 'C1209970710-ASF'],
    'PAULI': ['C1214354031-ASF', 'C1207038647-ASF'],
    'SLOPE': ['C1214408428-ASF', 'C1210599503-ASF'],
    'STOKES': ['C1214419355-ASF', 'C1210599673-ASF'],
    'L0B': [
        'C1261815274-ASFDEV', # RRSD BETA
        'C1273831320-ASF',
        'C3622228339-ASF',
        'C1261832497-ASFDEV', # Provisional
        'C1261832659-ASF',
        'C2853089814-ASF',
        'C1256358262-ASFDEV', # Validated
        'C1257349115-ASF',
        'C3622265756-ASF',
        'C1258125097-ASFDEV',
        'C1258836794-ASF',
        'C2887469134-ASF',
    ],
    'RRSD': [
        'C1261815274-ASFDEV', # RRSD BETA
        'C1273831320-ASF',
        'C3622228339-ASF',
        'C1261832497-ASFDEV', # Provisional
        'C1261832659-ASF',
        'C2853089814-ASF',
        'C1256358262-ASFDEV', # Validated
        'C1257349115-ASF',
        'C3622265756-ASF',
        
        'C1258125097-ASFDEV',
        'C1258836794-ASF',
        'C2887469134-ASF',
    ],
    'RSLC': [
            'C1261813489-ASFDEV', # Beta
            'C1273831203-ASF',
            'C2850225585-ASF',
            'C1261832868-ASFDEV', # Provisional
            'C1261833052-ASF',
            'C2853145197-ASF',
            'C1273095154-ASFDEV', # Validated
            'C1273831205-ASF',
            'C3622236985-ASF',
    ],
    'RIFG': [
            'C1261819086-ASFDEV', # Beta
            'C1273831200-ASF',
            'C2850234202-ASF',
            'C1261832940-ASFDEV', # Provisional
            'C1261833063-ASF',
            'C2853147928-ASF',
            'C1256381769-ASFDEV', # Validated
            'C1257349108-ASF',
            'C3622229381-ASF',
        ],
    'RUNW': [
        'C1261819098-ASFDEV', # Beta
        'C1261819121-ASF',
        'C2850235455-ASF',
        'C1261832990-ASFDEV', # Provisional
        'C1261833064-ASF',
        'C2853153429-ASF',
        'C1256420738-ASFDEV', # Validated
        'C1257349107-ASF',
        'C3622237369-ASF',
    ],
    'ROFF': [
        'C1261819110-ASFDEV', # Beta
        'C1261819145-ASF',
        'C2850237619-ASF',
        'C1261832993-ASFDEV', # Provisional
        'C1261833076-ASF',
        'C2853156054-ASF',
        'C1256411631-ASFDEV', # Validated
        'C1257349103-ASF',
        'C3622241997-ASF',
    ],
    'GCOV': [
        'C1261819211-ASFDEV', # Beta
        'C1273831195-ASF',
        'C3622214170-ASF',
        'C1261833026-ASFDEV', # Provisional
        'C1261846880-ASF',
        'C2854338529-ASF',
        'C1256477304-ASFDEV', # Validated
        'C1257349095-ASF',
        'C2727896018-ASF',
    ],
    'GOFF': [
        'C1261819233-ASFDEV', # Beta
        'C1261819281-ASF',
        'C2850263910-ASF',
        'C1261833027-ASFDEV', # Provisional
        'C1261846994-ASF',
        'C2854341702-ASF',
        'C1256479237-ASFDEV', # Validated
        'C1257349094-ASF',
        'C3622250865-ASF',
    ],
    'GSLC': [
        # 'NISAR_L2_GSLC_BETA_V1'
        'C1261819167-ASFDEV',
        'C1273831198-ASF',
        'C2850259510-ASF',
        # 'NISAR_L2_GSLC_PROVISIONAL_V1'
        'C1261833024-ASFDEV',
        'C1261833127-ASF',
        'C2854332392-ASF',
        # 'NISAR_L2_GSLC_V1'
        'C1256413628-ASFDEV',
        'C1257349102-ASF',
        'C3622244601-ASF',
    ],
    'GUNW': [
        # 'NISAR_L2_GUNW_BETA_V1':
        'C1261819168-ASFDEV',
        'C1261819270-ASF',
        'C2850261892-ASF',
        # 'NISAR_L2_GUNW_PROVISIONAL_V1'
        'C1261833025-ASFDEV',
        'C1261846741-ASF',
        'C2854335566-ASF',
        # 'NISAR_L2_GUNW_V1'
        'C1256432264-ASFDEV',
        'C1257349096-ASF',
        'C3622247503-ASF',
    ],
    'SME2': [
        'C1261819245-ASFDEV', # Beta
        'C1261819282-ASF',
        'C2850265000-ASF',
        'C1261833050-ASFDEV', # Provisional
        'C1261847095-ASF',
        'C2854344945-ASF',
        'C1256568692-ASFDEV', # Validated
        'C1257349093-ASF',
        'C3622248530-ASF',
    ],
}

# Helper Methods


def get_concept_id_alias(param_list: List[str], collections_dict: dict) -> List[str]:
    """
    param: param_list (List[str]): list of search values to alias
    param: collections_dict (dict): The search value to concept-id dictionary to read from

    returns List[str]: Returns a list of concept-ids
    that correspond to the given list of search values
    If any of the search values are not keys in the collections_dict,
    this will instead returns an empty list.
    """
    concept_id_aliases = []
    for param in param_list:
        if alias := collections_dict.get(param):
            concept_id_aliases.extend(alias)
        else:
            return []

    return concept_id_aliases


def get_dataset_concept_ids(datasets: List[str]) -> List[str]:
    """
    Returns concept-ids for provided dataset(s)
    If an invalid datset is provided a ValueError is raised

    :param `datasets` (`List[str]`): a list of datasets to grab concept-ids for
    :returns `List[str]`: the list of concept-ids associated with the given datasets
    """
    output = []
    for dataset in datasets:
        if collections_by_short_name := dataset_collections.get(dataset):
            for concept_ids in collections_by_short_name.values():
                output.extend(concept_ids)
        else:
            raise ValueError(
                f'Could not find dataset named "{dataset}" provided for dataset keyword.'
            )

    return output
