/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2024 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#ifndef MESHPROJECTION_H
#define MESHPROJECTION_H

#include <Action.h>

#include <MeshComponent.h>
#include <ImageComponent.h>
#include <Slice.h>
#include <InteractiveViewer.h>

//--vtk
// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkPolyDataMapper.h>
#include <CamiTKReEnableWarnings>

#include <vtkPlane.h>

/** Show the mesh contour in the slice viewers.
 *
 *  This action needs two input components: a mesh component and an image component.
 *
 *  The projection of a mesh is displayed as contour onto an image on each slice viewer (sagittal, coronal and axial)
 *  in respectively red, green and blue color.
 *
 */
class MeshProjection : public camitk::Action {
    Q_OBJECT

public:

    /// Default Constructor
    MeshProjection(camitk::ActionExtension*);

    /// Default Destructor
    virtual ~MeshProjection();

    /// manage property modification immediatly
    virtual bool event(QEvent* e);

    /// method called when the action when the action is triggered (i.e. started)
    virtual QWidget* getWidget();

public slots:
    /** this method is automatically called when the action is triggered.
      * Call getTargets() method to get the list of components to use.
      * \note getTargets() is automatically filtered so that it only contains compatible components,
      * i.e., instances of MeshProjection (or a subclass).
      */
    virtual camitk::Action::ApplyStatus apply();

    /// this slots is called when the user changes the selected slice to update the contour
    void updateCuttingPlane();

    /// remove the contour from the viewers if the mesh is closed / destroyed
    void hide();

private:
    /// current mesh component
    camitk::MeshComponent* meshToProject;

    /// current image component to project onto
    camitk::ImageComponent* targetImage;

    /// update the managed mesh and check the change in the image component using the corresponding property
    void updateComponents(camitk::MeshComponent*);

    // used to map item id with component for the "Component List" property
    QList<camitk::ImageComponent*> imageComponentList;

    /// show/hide the contour in a viewer
    void updateVisibility();

    /// the contour mappers (input for both 3D and 2D actors)
    QMap<camitk::Slice::SliceOrientation, vtkSmartPointer<vtkPolyDataMapper>> cutterMapperMap;

    /// the mesh contours in the 3D view (where the mesh actually is), one for each orientation
    QMap<camitk::Slice::SliceOrientation, vtkSmartPointer<vtkActor>> contourActorMap;

    /// the mesh contours in the 2D views (with the proper transform that makes it appear in the slice viewer)
    QMap<camitk::Slice::SliceOrientation, vtkSmartPointer<vtkActor>> contourActorIn2DViewerMap;

    /// the cutting planes: the contour is the intersection between these planes and the mesh
    QMap<camitk::Slice::SliceOrientation, vtkSmartPointer<vtkPlane>> cuttingPlaneMap;

    /// get the dimension index corresponding to the current orientation
    /// 0 is the index for sagittal dimension, 1 for the coronal orientation and 2 is for the axial orientation
    int getOrientationIndex(camitk::Slice::SliceOrientation);

    /// get the voxel slice in the correct orientation
    double getVoxelSize(camitk::Slice::SliceOrientation);

    /// get the slice viewer corresponding to the orientation
    camitk::InteractiveViewer* getViewer(camitk::Slice::SliceOrientation);

    /// create an actor for the contour. The color of the new actor depends on the current orientation
    vtkSmartPointer<vtkActor> getNewActor(camitk::Slice::SliceOrientation);

    /// create a plane for a given orientation
    vtkSmartPointer<vtkPlane> getNewPlane(camitk::Slice::SliceOrientation);

    /// update the cutting plane position depending on the current slice and the given orientation
    void updateCuttingPlane(camitk::Slice::SliceOrientation);

    /// update the contour line width using the corresponding property value
    void updateContourLineWidth();
};

#endif // MESHPROJECTION_H

