/*
 *	Ported to Linux's Second Extended File System as part of the
 *	dump and restore backup suit
 *	Remy Card <card@Linux.EU.Org>, 1994-1997
 *	Stelian Pop <stelian@popies.net>, 1999-2000
 *	Stelian Pop <stelian@popies.net> - Alcôve <www.alcove.com>, 2000-2002
 */

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>
#include <sys/types.h>
#include <errno.h>
#include <compaterr.h>
#include <system.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/param.h>
#include <sys/file.h>
#include <sys/mtio.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <sys/sysmacros.h>
#include <sys/time.h>
#include <time.h>
#ifdef HAVE_EXT2FS_EXT2_FS_H
#include <ext2fs/ext2_fs.h>
#else
#include <linux/ext2_fs.h>
#endif
#include <ext2fs/ext2fs.h>
#include <bsdcompat.h>
#ifdef DUMP_MACOSX
#include "darwin.h"
#endif
#include <protocols/dumprestore.h>

#ifdef HAVE_ZLIB
#include <zlib.h>
#define HAVE_COMPRESSION 1
#endif /* HAVE_ZLIB */

#ifdef HAVE_BZLIB
#include <bzlib.h>
#define HAVE_COMPRESSION 1
#endif /* HAVE_BZLIB */

#ifdef HAVE_LZO
#include <lzo/lzo1x.h>
#define HAVE_COMPRESSION 1
#endif /* HAVE_LZO */

#include "restore.h"
#include "extern.h"
#include "pathnames.h"
#include "readtape.h"

#ifdef USE_QFA
int		noresyncmesg = 0;
#endif /* USE_QFA */
static long	fssize = MAXBSIZE;
int		pipein = 0;
static char	magtape[MAXPATHLEN];
static char	magtapeprefix[MAXPATHLEN];
static int	blkcnt;
static int	numtrec;
static char	*tapebuf;		/* input buffer for read */
static int	bufsize;		/* buffer size without prefix */
static char	*tbufptr = NULL;	/* active tape buffer */
#ifdef HAVE_COMPRESSION
static void	*comprbuf;		/* uncompress work buf */
static size_t	comprlen;		/* size including prefix */
#endif
static union	u_spcl endoftapemark;
static long long	blksread;		/* blocks read since last header */
static long long	tpblksread = 0;		/* TP_BSIZE blocks read */
static long	tapesread;
static sigjmp_buf	restart;
static int	gettingfile = 0;	/* restart has a valid frame */
char		*host = NULL;

static int	ofile;
char inline_data[INLINE_DATA_MAX_INODE_SIZE + XATTR_MAXSIZE];

static char	*map;
static char	lnkbuf[MAXPATHLEN + 1];
static int	pathlen;

int		oldinofmt;	/* old inode format conversion required */
int		Bcvt;		/* Swap Bytes (for CCI or sun) */

#define	FLUSHTAPEBUF()	blkcnt = ntrec + 1

static void	 accthdr (struct s_spcl *);
static int	 checksum (int *);
static void	 findinode (struct s_spcl *);
static void	 findtapeblksize (void);
static int	 gethead (struct s_spcl *);
int	 converthead (struct s_spcl *);
static void	 readtape (char *);
static void	 setdumpnum (void);
#ifdef DUMP_MACOSX
static void	 xtrfilefinderinfo (const char *, off_t, size_t);
#endif

static u_int	 swabi (u_int);
#if 0
static u_long	 swabl (u_long);
#endif
static u_char	*swab64 (u_char *, int);
static u_char	*swab32 (u_char *, int);
static u_char	*swab16 (u_char *, int);
static void	 terminateinput (void);
static void	 xtrfile (const char *, off_t, size_t);
static void	 xtrlnkfile (const char *, off_t, size_t);
static void	 xtrlnkskip (const char *, off_t, size_t);
static void	 xtrmap (const char *, off_t, size_t);
static void	 xtrmapskip (const char *, off_t, size_t);
static void	 xtrskip (const char *, off_t, size_t);
static void	 xtrxattr (const char *, off_t, size_t);
static void	 setmagtapein (void);
static int	 extractattr (char *);
static void	 compareattr (char *);

#ifdef HAVE_COMPRESSION
static void	newcomprbuf (int);
#endif
static char	*decompress_tapebuf (struct tapebuf *, int);
static void	msg_read_error (char *);
#define PREFIXSIZE	sizeof(struct tapebuf)

#define COMPARE_ONTHEFLY 1

#if COMPARE_ONTHEFLY
static int	ifile;		/* input file for compare */
static int	cmperror;	/* compare error */
static void	xtrcmpfile (const char *, off_t, size_t);
static void	xtrcmpskip (const char *, off_t, size_t);
#endif

static int readmapflag;
static int readingmaps;		/* set to 1 while reading the maps */

static char xattrbuf[XATTR_MAXSIZE];
static int xattrlen;

#ifdef DUMP_MACOSX
static DumpFinderInfo	gFndrInfo;
#endif

/*
 * Set up an input source. This is called from main.c before setup() is.
 */
void
setinput(char *source)
{
	int i;
	char *n;

	FLUSHTAPEBUF();
	if (bflag)
		newtapebuf(ntrec);
	else
		newtapebuf(NTREC > HIGHDENSITYTREC ? NTREC : HIGHDENSITYTREC);
	terminal = stdin;

#ifdef RRESTORE
	if ((n = strchr(source, ':'))) {
		for (i = 0; i < (n - source); i++) {
			if (source[i] == '/')
				break;
		}
		if (source[i] != '/') {
			host = source;
			source = strchr(host, ':');
			*source++ = '\0';
			if (rmthost(host) == 0)
				exit(1);
		}
	} else
#endif
	{
	if (strcmp(source, "-") == 0) {
		/*
		 * Since input is coming from a pipe we must establish
		 * our own connection to the terminal.
		 */
		terminal = fopen(_PATH_TTY, "r");
		if (terminal == NULL) {
			warn("cannot open %s", _PATH_TTY);
			terminal = fopen(_PATH_DEVNULL, "r");
			if (terminal == NULL)
				err(1, "cannot open %s", _PATH_DEVNULL);
		}
		pipein++;
	}
	}
	/* no longer need or want root privileges */
	if (setuid(getuid()))
		err(1, "cannot setuid");
	if (Mflag) {
		strncpy(magtapeprefix, source, MAXPATHLEN);
		magtapeprefix[MAXPATHLEN-1] = '\0';
		snprintf(magtape, MAXPATHLEN, "%s%03d", source, 1);
	}
	else
		strncpy(magtape, source, MAXPATHLEN);
	magtape[MAXPATHLEN - 1] = '\0';
}

void
newtapebuf(long size)
{
	static int tapebufsize = -1;

	ntrec = size;
	bufsize = ntrec * TP_BSIZE;
	if (size <= tapebufsize)
		return;
	if (tapebuf != NULL)
		free(tapebuf);
	tapebuf = malloc(size * TP_BSIZE + sizeof(struct tapebuf));
	if (tapebuf == NULL)
		errx(1, "Cannot allocate space for tape buffer");
	tapebufsize = size;
}

#ifdef HAVE_COMPRESSION
static void
newcomprbuf(int size)
{
	size_t buf_size = (size+1) * TP_BSIZE + sizeof(struct tapebuf);
	if (buf_size <= comprlen)
		return;
	comprlen = buf_size;
	if (comprbuf != NULL)
		free(comprbuf);
	comprbuf = malloc(comprlen);
	if (comprbuf == NULL)
		errx(1, "Cannot allocate space for decompress buffer");
}
#endif /* HAVE_COMPRESSION */

/*
 * Verify that the tape drive can be accessed and
 * that it actually is a dump tape.
 */
void
setup(void)
{
	int i, j, *ip, bot_code;
	struct stat stbuf;
	char *temptape;

	Vprintf(stdout, "Verify tape and initialize maps\n");
	if (Afile == NULL && bot_script) {
		msg("Launching %s\n", bot_script);
		bot_code = system_command(bot_script, magtape, 1);
		if (bot_code != 0 && bot_code != 1) {
			msg("Restore aborted by the beginning of tape script\n");
			exit(1);
		}
	}

	if (Afile)
		temptape = Afile;
	else
		temptape = magtape;

	if (open_tape(!Afile && !!host, temptape) < 0)
		err(1, "%s", temptape);

	if (!Afile) {
		volno = 1;
		setmagtapein();
		setdumpnum();
	}
#if defined(HAVE_LZO)
	if (lzo_init() != LZO_E_OK) {
		msg("internal error - lzo_init failed \n");
		exit(1);
	}
#endif
	FLUSHTAPEBUF();
	findtapeblksize();
	cvtflag = 0;
	if (gethead(&spcl) == FAIL) {
		blkcnt--; /* push back this block */
		blksread--;
		tpblksread--;
		cvtflag++;
		if (gethead(&spcl) == FAIL)
			errx(1, "Tape is not a dump tape");
		fprintf(stderr, "Converting to new file system format.\n");
	}

	if (pipein) {
		endoftapemark.s_spcl.c_magic = cvtflag ? OFS_MAGIC : NFS_MAGIC;
		endoftapemark.s_spcl.c_type = TS_END;
		ip = (int *)&endoftapemark;
		j = sizeof(union u_spcl) / sizeof(int);
		i = 0;
		do
			i += *ip++;
		while (--j);
		endoftapemark.s_spcl.c_checksum = CHECKSUM - i;
	}
	if (vflag || command == 't' || command == 'C')
		printdumpinfo();
#ifdef USE_QFA
	if (tapeposflag && (uint64_t)spcl.c_date != qfadumpdate)
		errx(1, "different QFA/dumpdates detected\n");
#endif
	if (filesys[0] == '\0') {
		char *dirptr;
		strncpy(filesys, spcl.c_filesys, NAMELEN);
		filesys[NAMELEN - 1] = '\0';
		dirptr = strstr(filesys, " (dir");
		if (dirptr != NULL)
			*dirptr = '\0';
	}
	dumptime = spcl.c_ddate;
	dumpdate = spcl.c_date;
	if (stat(".", &stbuf) < 0)
		err(1, "cannot stat .");
	if (stbuf.st_blksize > 0 && stbuf.st_blksize < TP_BSIZE )
		fssize = TP_BSIZE;
	if (stbuf.st_blksize >= TP_BSIZE && stbuf.st_blksize <= MAXBSIZE)
		fssize = stbuf.st_blksize;
	if (fssize % TP_BSIZE) {
		long realfssize = fssize;
		warnx("file system block size(%ld) is not a multiple of TP_BSIZE(%d)\n", fssize, TP_BSIZE);
		while (fssize % TP_BSIZE && fssize < MAXBSIZE)
			fssize = (fssize+realfssize < MAXBSIZE)?fssize+realfssize:MAXBSIZE;
		if (fssize % realfssize)
			warnx("No common multiple of %ld and %d available.", realfssize, TP_BSIZE);
		warnx("Using %ld to write blocks to disk instead\n", fssize);
	}
	if (spcl.c_volume != 1)
		errx(1, "Tape is not volume 1 of the dump");
	if (gethead(&spcl) == FAIL) {
		Dprintf(stdout, "header read failed at %lld blocks\n", blksread);
		panic("no header after volume mark!\n");
	}
	readingmaps = 1;
	findinode(&spcl);
	if (spcl.c_type != TS_CLRI)
		errx(1, "Cannot find file removal list");
	maxino = (spcl.c_count * TP_BSIZE * NBBY) + 1;
	map = calloc(1, howmany(maxino, NBBY));
	if (map == NULL)
		errx(1, "no memory for active inode map");
	usedinomap = map;
	curfile.action = USING;
	getfile(xtrmap, xtrmapskip);
	while (spcl.c_type == TS_ADDR) {
		/* Recompute maxino and the map */
		dump_ino_t oldmaxino = maxino;
		maxino += (spcl.c_count * TP_BSIZE * NBBY) + 1;
		resizemaps(oldmaxino, maxino);
		map = usedinomap;

		spcl.c_dinode.di_size = spcl.c_count * TP_BSIZE;
		getfile(xtrmap, xtrmapskip);
	}
	Dprintf(stdout, "maxino = %u\n", maxino);
	if (spcl.c_type != TS_BITS) {
		if (spcl.c_type == TS_END) {
			msg("Cannot find file dump list, assuming empty tape\n");
			exit(0);
		}
		errx(1, "Cannot find file dump list");
	}
	map = calloc(1, howmany(maxino, NBBY));
	if (map == (char *)NULL)
		errx(1, "no memory for file dump list");
	dumpmap = map;
	curfile.action = USING;
	getfile(xtrmap, xtrmapskip);
	while (spcl.c_type == TS_ADDR) {
		spcl.c_dinode.di_size = spcl.c_count * TP_BSIZE;
		getfile(xtrmap, xtrmapskip);
	}
	/*
	 * If there may be whiteout entries on the tape, pretend that the
	 * whiteout inode exists, so that the whiteout entries can be
	 * extracted.
	 */
	if (oldinofmt == 0)
		SETINO(WINO, dumpmap);
	readingmaps = 0;
	findinode(&spcl);
}

/*
 * Prompt user to load a new dump volume.
 * "Nextvol" is the next suggested volume to use.
 * This suggested volume is enforced when doing full
 * or incremental restores, but can be overridden by
 * the user when only extracting a subset of the files.
 */
void
getvol(long nextvol)
{
	long newvol = 0, wantnext = 0, i;
	long long saved_blksread = 0, saved_tpblksread = 0;
	union u_spcl tmpspcl;
#	define tmpbuf tmpspcl.s_spcl
	char buf[TP_BSIZE];
	int haderror = 0, bot_code = 1;

	if (nextvol == 1) {
		tapesread = 0;
		gettingfile = 0;
		blksread = 0;
	}
	if (pipein) {
		if (nextvol != 1)
			panic("Changing volumes on pipe input?\n");
		if (volno == 1)
			return;
		goto gethdr;
	}
	saved_blksread = blksread;
	saved_tpblksread = tpblksread;
again:
	if (pipein)
		exit(1); /* pipes do not get a second chance */
	if (aflag || curfile.action != SKIP) {
		newvol = nextvol;
		wantnext = 1;
	} else {
		newvol = 0;
		wantnext = 0;
	}
	while (newvol <= 0) {
		if (tapesread == 0) {
			fprintf(stderr, "%s%s%s%s%s",
			    "You have not read any volumes yet.\n",
			    "Unless you know which volume your",
			    " file(s) are on you should start\n",
			    "with the last volume and work",
			    " towards the first.\n");
		} else {
			fprintf(stderr, "You have read volumes");
			strcpy(buf, ": ");
			for (i = 1; i < 32; i++)
				if (tapesread & (1 << i)) {
					fprintf(stderr, "%s%ld", buf, i);
					strcpy(buf, ", ");
				}
			fprintf(stderr, "\n");
		}
		do	{
			fprintf(stderr, "Specify next volume # (none if no more volumes): ");
			(void) fflush(stderr);
			if (!fgets(buf, TP_BSIZE, terminal))
				break;
		} while (!feof(terminal) && buf[0] == '\n');
		if (feof(terminal))
			exit(1);
		if (!strcmp(buf, "none\n")) {
			terminateinput();
			return;
		}
		newvol = atoi(buf);
		if (newvol <= 0) {
			fprintf(stderr,
			    "Volume numbers are positive numerics\n");
		}
	}
	if (newvol == volno) {
		tapesread |= 1 << volno;
		return;
	}
	close_tape();

	/*
	 * if using an archive file, reset its name so readtape()
	 * could properly use remote access.
	 */
	Afile = NULL;

	if (Mflag) {
		size_t len = snprintf(magtape, MAXPATHLEN, "%s%03ld", magtapeprefix, newvol);
		if (len >= MAXPATHLEN)
			err(1, "%s%03ld is more than %d characters", magtapeprefix, newvol, MAXPATHLEN);
		magtape[MAXPATHLEN - 1] = '\0';
	}
	if (bot_script && !haderror) {
		msg("Launching %s\n", bot_script);
		bot_code = system_command(bot_script, magtape, newvol);
		if (bot_code != 0 && bot_code != 1) {
			msg("Restore aborted by the beginning of tape script\n");
			exit(1);
		}
	}
	if (haderror || (bot_code && !Mflag)) {
		haderror = 0;
		if (compare_errors)
			fprintf(stderr, "%d compare errors so far\n", compare_errors);
		fprintf(stderr, "Mount tape volume %ld\n", newvol);
		fprintf(stderr, "Enter ``none'' if there are no more tapes\n");
		fprintf(stderr, "otherwise enter tape name (default: %s) ", magtape);
		(void) fflush(stderr);
		if (!fgets(buf, TP_BSIZE, terminal))
			exit(1);
		if (feof(terminal))
			exit(1);
		if (!strcmp(buf, "none\n")) {
			terminateinput();
			return;
		}
		if (buf[0] != '\n') {
			char *pos;
			(void) strncpy(magtape, buf, sizeof(magtape));
			magtape[sizeof(magtape) - 1] = '\0';
			if ((pos = strchr(magtape, '\n')))
				magtape[pos - magtape] = '\0';
		}
	}
	if(open_tape(!!host, magtape) == -1) {
		fprintf(stderr, "Cannot open %s\n", magtape);
		volno = -1;
		haderror = 1;
		goto again;
	}
gethdr:
	setmagtapein();
	volno = newvol;
	setdumpnum();
	FLUSHTAPEBUF();
	findtapeblksize();
	if (gethead(&tmpbuf) == FAIL) {
		Dprintf(stdout, "header read failed at %lld blocks\n", blksread);
		fprintf(stderr, "tape is not dump tape\n");
		volno = 0;
		haderror = 1;
		blksread = saved_blksread;
		tpblksread = saved_tpblksread;
		goto again;
	}
	if (tmpbuf.c_volume != volno) {
		fprintf(stderr, "Wrong volume (%d)\n", tmpbuf.c_volume);
		volno = 0;
		haderror = 1;
		blksread = saved_blksread;
		tpblksread = saved_tpblksread;
		goto again;
	}
	if (tmpbuf.c_date != dumpdate || tmpbuf.c_ddate != dumptime) {
		fprintf(stderr, "Wrong dump date\n\tgot: %s",
			ctime4(&tmpbuf.c_date));
		fprintf(stderr, "\twanted: %s", ctime(&dumpdate));
		volno = 0;
		haderror = 1;
		blksread = saved_blksread;
		tpblksread = saved_tpblksread;
		goto again;
	}
	tapesread |= 1 << volno;
	/*
	 * If continuing from the previous volume, skip over any
	 * blocks read already at the end of the previous volume.
	 *
	 * If coming to this volume at random, skip to the beginning
	 * of the next record.
	 */
	Dprintf(stdout, "read %lld recs, tape starts with %lld\n",
		tpblksread - 1, u_spcl_c_firstrec(&tmpbuf));
	if (tmpbuf.c_type == TS_TAPE && (tmpbuf.c_flags & DR_NEWHEADER)) {
		if (!wantnext) {
			tpblksread = u_spcl_c_firstrec(&tmpbuf) + 1;
			for (i = tmpbuf.c_count; i > 0; i--)
				readtape(buf);
		} else if (u_spcl_c_firstrec(&tmpbuf) > 0 &&
			   u_spcl_c_firstrec(&tmpbuf) < tpblksread - 1) {
			/*
			 * -1 since we've read the volume header
			 */
			i = tpblksread - u_spcl_c_firstrec(&tmpbuf) - 1;
			Dprintf(stderr, "Skipping %ld duplicate record%s.\n",
				i, i > 1 ? "s" : "");
			while (--i >= 0)
				readtape(buf);
		}
	}
	if (curfile.action == USING) {
		if (volno == 1)
			panic("active file into volume 1\n");
		return;
	}
	/*
	 * Skip up to the beginning of the next record
	 */
	if (tmpbuf.c_type == TS_TAPE && (tmpbuf.c_flags & DR_NEWHEADER))
		for (i = tmpbuf.c_count; i > 0; i--)
			readtape(buf);
	(void) gethead(&spcl);
	findinode(&spcl);
	if (gettingfile) {
		gettingfile = 0;
		siglongjmp(restart, 1);
	}
}

/*
 * Handle unexpected EOF.
 */
static void
terminateinput(void)
{

	if (gettingfile && curfile.action == USING) {
		printf("Warning: %s %s\n",
		    "End-of-input encountered while extracting", curfile.name);
	}
	curfile.name = "<name unknown>";
	curfile.action = UNKNOWN;
	curfile.dip = NULL;
	curfile.ino = maxino;
	if (gettingfile) {
		gettingfile = 0;
		siglongjmp(restart, 1);
	}
}

/*
 * handle multiple dumps per tape by skipping forward to the
 * appropriate one.
 */
static void
setdumpnum(void)
{
	if (dumpnum == 1 || volno != 1)
		return;
	if (pipein)
		errx(1, "Cannot have multiple dumps on pipe input");

	if (ioctl_tape(MTFSF, dumpnum - 1) < 0) {
		fprintf(stderr, "ioctl MTFSF: %s\n", strerror(errno));
		exit(1);
	}
}

void
printdumpinfo(void)
{
	fprintf(stdout, "Dump   date: %s", ctime4(&spcl.c_date));
	fprintf(stdout, "Dumped from: %s",
	    (spcl.c_ddate == 0) ? "the epoch\n" : ctime4(&spcl.c_ddate));
	if (spcl.c_host[0] == '\0')
		return;
	fprintf(stdout, "Level %d dump of %s on %s:%s\n",
		spcl.c_level, spcl.c_filesys, spcl.c_host, spcl.c_dev);
	fprintf(stdout, "Label: %s\n", spcl.c_label);
}

void
printvolinfo(void)
{
	int i;

	if (volinfo[1] == ROOTINO) {
		printf("Starting inode numbers by volume:\n");
		for (i = 1; i < (int)TP_NINOS && volinfo[i] != 0; ++i)
			printf("\tVolume %d: %u\n", i, volinfo[i]);
	}
}


int
extractfile(struct entry *ep, int doremove)
{
	unsigned int flags;
	mode_t mode;
	struct timeval timep[2];
	char *name = myname(ep);

	if (Nflag) {
		skipfile();
		return (GOOD);
	}
	/* If removal is requested (-r mode) do remove it unless
	 * we are extracting a metadata only inode */
	if (spcl.c_flags & DR_METAONLY) {
		Vprintf(stdout, "file %s is metadata only\n", name);
	}
	else {
		if (doremove) {
			removeleaf(ep);
			ep->e_flags &= ~REMOVED;
		}
	}

	curfile.name = name;
	curfile.action = USING;
	timep[0].tv_sec = curfile.dip->di_atime.tv_sec;
	timep[0].tv_usec = curfile.dip->di_atime.tv_usec;
	timep[1].tv_sec = curfile.dip->di_mtime.tv_sec;
	timep[1].tv_usec = curfile.dip->di_mtime.tv_usec;
	mode = curfile.dip->di_mode;
	flags = curfile.dip->di_flags;
	switch (mode & IFMT) {

	default:
		fprintf(stderr, "%s: unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFSOCK:
	{
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;

		Vprintf(stdout, "extract socket %s\n", name);
		skipfile();
		if (Nflag)
			return (GOOD);
		if (! (spcl.c_flags & DR_METAONLY)) {
			if (mknod(name, S_IFSOCK | 0600, 0) < 0) {
				warn("%s: cannot create socket", name);
				return (FAIL);
			}
		}
		if (chown(name, luid, lgid) < 0)
			warn("%s: chown", name);
		if (chmod(name, mode) < 0)
			warn("%s: chmod", name);
		extractattr(name);
		utimes(name, timep);
		if (flags)
			(void) lsetflags(name, flags);
		return (GOOD);
	}

	case IFDIR:
	{
		int ret;
		if (mflag) {
			if (ep == NULL || ep->e_flags & EXTRACT)
				panic("unextracted directory %s\n", name);
			skipfile();
			return (GOOD);
		}
		Vprintf(stdout, "extract dir %s\n", name);
		ret = genliteraldir(name, curfile.ino);
		extractattr(name);
		return ret;
	}

	case IFLNK:
	{
#ifdef HAVE_LCHOWN
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;
#endif
#ifdef HAVE_UTIMENSAT
		struct timespec times[2];

		times[0].tv_sec  =  timep[0].tv_sec;
		times[0].tv_nsec =  timep[0].tv_usec * 1000;
		times[1].tv_sec  =  timep[1].tv_sec;
		times[1].tv_nsec =  timep[1].tv_usec * 1000;
#endif

		if (! (spcl.c_flags & DR_METAONLY)) {
			lnkbuf[0] = '\0';
			pathlen = 0;
			getfile(xtrlnkfile, xtrlnkskip);
			if (pathlen == 0) {
				Vprintf(stdout,
				    "%s: zero length symbolic link (ignored)\n", name);
				return (GOOD);
			}
			if (linkit(lnkbuf, name, SYMLINK) == FAIL)
				return (FAIL);
		}
		else
			skipfile();

#ifdef HAVE_LCHOWN
		if (lchown(name, luid, lgid) < 0)
			warn("%s: lchown", name);
#endif
#ifdef HAVE_UTIMENSAT
		if (utimensat(AT_FDCWD, name, times, AT_SYMLINK_NOFOLLOW) < 0)
			 warn("%s: file timestamp update failed", name);
#endif
		extractattr(name);
		return (GOOD);
	}

	case IFIFO:
	{
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;

		Vprintf(stdout, "extract fifo %s\n", name);
		skipfile();
		if (Nflag)
			return (GOOD);
		if (! (spcl.c_flags & DR_METAONLY)) {
			if (uflag && !Nflag)
				(void)unlink(name);
			if (mkfifo(name, mode) < 0) {
				warn("%s: cannot create fifo", name);
				return (FAIL);
			}
		}
		if (chown(name, luid, lgid) < 0)
			warn("%s: chown", name);
		if (chmod(name, mode) < 0)
			warn("%s: chmod", name);
		extractattr(name);
		utimes(name, timep);
		if (flags)
			(void) lsetflags(name, flags);
		return (GOOD);
	}
	case IFCHR:
	case IFBLK:
	{
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;
		int lrdev = (int)curfile.dip->di_rdev;

		Vprintf(stdout, "extract special file %s\n", name);
		skipfile();
		if (Nflag)
			return (GOOD);
		if (! (spcl.c_flags & DR_METAONLY)) {
			if (uflag)
				(void)unlink(name);
			if (mknod(name, mode, lrdev) < 0) {
				warn("%s: cannot create special file", name);
				return (FAIL);
			}
		}
		if (chown(name, luid, lgid) < 0)
			warn("%s: chown", name);
		if (chmod(name, mode) < 0)
			warn("%s: chmod", name);
		extractattr(name);
		utimes(name, timep);
		if (flags)
			{
			warn("%s: lsetflags called on a special file", name);
			(void) lsetflags(name, flags);
			}
		return (GOOD);
	}
	case IFREG:
	{
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;

		Vprintf(stdout, "extract file %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}
		if (! (spcl.c_flags & DR_METAONLY)) {
			if (uflag)
				(void)unlink(name);
			if ((ofile = open(name, O_WRONLY | O_CREAT | O_TRUNC,
			    0666)) < 0) {
				warn("%s: cannot create file", name);
				skipfile();
				return (FAIL);
			}
			if (curfile.dip->di_flags & EXT4_INLINE_DATA_FL) {
				uint64_t inline_filesize = curfile.dip->di_size>sizeof(inline_data)?sizeof(inline_data):curfile.dip->di_size;
				memcpy(inline_data, curfile.dip->di_db, INLINE_DATA_MAX_INODE_SIZE);
				getfile(xtrfile, xtrnull);
				extractattr(name);
				xtrfile(inline_data, 0, inline_filesize);
			} else {
				getfile(xtrfile, xtrskip);
				extractattr(name);
			}
			(void) close(ofile);
		}
		else {
			skipfile();
			extractattr(name);
		}
		if (chown(name, luid, lgid) < 0)
			warn("%s: chown", name);
		if (chmod(name, mode) < 0)
			warn("%s: chmod", name);
		utimes(name, timep);
		if (flags)
			(void) lsetflags(name, flags);
		return (GOOD);
	}
	}
	/* NOTREACHED */
}

static int
extractattr(char *path)
{
	while (spcl.c_flags & DR_EXTATTRIBUTES) {
		switch (spcl.c_extattributes) {
		case EXT_MACOSFNDRINFO:
#ifdef DUMP_MACOSX
			(void)extractfinderinfoufs(path);
#else
			msg("MacOSX not supported in this version, skipping\n");
			skipfile();
#endif
			break;
		case EXT_MACOSRESFORK:
#ifdef DUMP_MACOSX
			(void)extractresourceufs(path);
#else
			msg("MacOSX not supported in this version, skipping\n");
			skipfile();
#endif
			break;
		case EXT_XATTR: {
			char xattr[XATTR_MAXSIZE];

			if (readxattr(xattr) == GOOD)
				xattr_extract(path, xattr);
			break;
		}
		default:
			msg("unexpected inode extension %ld, skipping\n", spcl.c_extattributes);
			skipfile();
			break;
		}
	}
	return GOOD;
}

#ifdef DUMP_MACOSX
int
extractfinderinfoufs(char *name)
{
	int err;
	char			oFileRsrc[MAXPATHLEN];
	int flags;
	mode_t mode;
	struct timeval timep[2];
	u_int32_t	uid;
	u_int32_t	gid;
	char	path[MAXPATHLEN], fname[MAXPATHLEN];

	curfile.name = name;
	curfile.action = USING;
	timep[0].tv_sec = curfile.dip->di_atime.tv_sec;
	timep[0].tv_usec = curfile.dip->di_atime.tv_usec;
	timep[1].tv_sec = curfile.dip->di_mtime.tv_sec;
	timep[1].tv_usec = curfile.dip->di_mtime.tv_usec;
	mode = curfile.dip->di_mode;
	flags = curfile.dip->di_flags;
	uid = curfile.dip->di_uid;
	gid =  curfile.dip->di_gid;

	switch (mode & IFMT) {

	default:
		fprintf(stderr, "%s: (extr. finfoufs) unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFDIR:
		fprintf(stderr, "%s: (extr. finfoufs[IFDIR]) unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFLNK:
		skipfile();
		return (GOOD);

	case IFREG:
		Vprintf(stdout, "extract finderinfo file %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}
		getfile(xtrfilefinderinfo, xtrskip);

		GetPathFile(name, path, fname);
		strcpy(oFileRsrc, path);
		strcat(oFileRsrc, "._");
		strcat(oFileRsrc, fname);

		if ((err = CreateAppleDoubleFileRes(oFileRsrc, &gFndrInfo.fndrinfo,
				mode, flags, timep, uid, gid)) != 0) {
			fprintf(stderr, "%s: cannot create finderinfo: %s\n",
			name, strerror(errno));
			skipfile();
			return (FAIL);
		}
		return (GOOD);
	}
	/* NOTREACHED */
}


int
extractresourceufs(char *name)
{
	char			oFileRsrc[MAXPATHLEN];
	int flags;
	mode_t mode;
	struct timeval timep[2];
	char	path[MAXPATHLEN], fname[MAXPATHLEN];
	ASDHeaderPtr	hp;
	ASDEntryPtr	ep;
	u_long	loff;
	u_int32_t	uid;
	u_int32_t	gid;
	u_int64_t	di_size;
	char		*p;
	char		buf[1024];

	curfile.name = name;
	curfile.action = USING;
	timep[0].tv_sec = curfile.dip->di_atime.tv_sec;
	timep[0].tv_usec = curfile.dip->di_atime.tv_usec;
	timep[1].tv_sec = curfile.dip->di_mtime.tv_sec;
	timep[1].tv_usec = curfile.dip->di_mtime.tv_usec;
	mode = curfile.dip->di_mode;
	flags = curfile.dip->di_flags;
	uid = curfile.dip->di_uid;
	gid =  curfile.dip->di_gid;
	di_size = curfile.dip->di_size;

	switch (mode & IFMT) {

	default:
		fprintf(stderr, "%s: (extr. resufs) unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFDIR:
		fprintf(stderr, "%s: (extr. resufs [IFDIR]) unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFLNK:
		skipfile();
		return (GOOD);

	case IFREG:
		Vprintf(stdout, "extract resource file %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}

		GetPathFile(name, path, fname);
		strcpy(oFileRsrc, path);
		strcat(oFileRsrc, "._");
		strcat(oFileRsrc, fname);

		if ((ofile = open(oFileRsrc, O_RDONLY, 0)) < 0) {
			fprintf(stderr, "%s: cannot read finderinfo: %s\n",
			    name, strerror(errno));
			skipfile();
			return (FAIL);
		}
		read(ofile, buf, 70);
		(void) close(ofile);
		p = buf;
		hp = (ASDHeaderPtr)p;
		/* the header */
		hp->entries++;
		p += sizeof(ASDHeader) - CORRECT;
		ep = (ASDEntryPtr)p;
		/* the finderinfo entry */
		ep->offset += sizeof(ASDEntry);
		loff = ep->offset;

		p += sizeof(ASDEntry);
		/* the finderinfo data */
		bcopy(p, p + sizeof(ASDEntry), INFOLEN);
		ep = (ASDEntryPtr)p;
		/* the new resourcefork entry */
		ep->entryID = EntryRSRCFork;
		ep->offset = loff + INFOLEN;
		ep->len = di_size;
		/* write the new appledouble entries to the file */
		if ((ofile = open(oFileRsrc, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0) {
			fprintf(stderr, "%s: cannot create resource file: %s\n",
			    name, strerror(errno));
			skipfile();
			return (FAIL);
		}
		write(ofile, buf, 70 + sizeof(ASDEntry));
		/* and add the resource data from tape */
		getfile(xtrfile, xtrskip);

		if (fchown(ofile, uid, gid) < 0)
			warn("%s: fchown", name);
		if (fchmod(ofile, mode) < 0)
			warn("%s: fchmod", name);
		(void) close(ofile);
		utimes(oFileRsrc, timep);
		(void) lsetflags(oFileRsrc, flags);
		return (GOOD);
	}
	/* NOTREACHED */
}
#endif /* DUMP_MACOSX */

int
readxattr(char *buffer)
{
	if (dflag)
		msg("reading EA data for inode %lu\n", curfile.ino);

	curfile.name = "EA block";
	curfile.action = USING;
	if (curfile.dip->di_size > XATTR_MAXSIZE) {
		fprintf(stderr, "EA size too big (%" PRIu64 ")", curfile.dip->di_size);
		skipfile();
		return (FAIL);
	}

	memset(xattrbuf, 0, XATTR_MAXSIZE);
	xattrlen = 0;

	/*
	 * ugly hack: cope with invalid spcl.c_addr[] records written by
	 * old versions of dump (versions 0.4b40 through 0.4b42)
	 * There are two different ways there can be a zero in c_addr: this
	 * bug, in which case the only other possible value is 1 or v0.4b49 and
	 * above where the previous byte must have the msb set (and, for ext4,
	 * this is currently impossible as it would require >64K disk block
	 * size and/or >64K of EA data)
	 */
	for(int i = 0; i<spcl.c_count; ++i)
		if(spcl.c_addr[0] >= 128)
			break;
		else if(spcl.c_addr[0] == 0) {
			msg("Fixing up invalid c_addr record for xattrs (v0.4b40-0.4b42 bug)\n");
			for(int i = 0; i<spcl.c_count; ++i)
				spcl.c_addr[i] = 1;
			break;
		}

	getfile(xtrxattr, xtrnull);

	memcpy(buffer, xattrbuf, XATTR_MAXSIZE);

	return (GOOD);
}

/*
 * skip over bit maps on the tape
 */
void
skipmaps(void)
{

	while (spcl.c_type == TS_BITS || spcl.c_type == TS_CLRI)
		skipfile();
}

/*
 * skip over a file on the tape
 */
void
skipfile(void)
{

	curfile.action = SKIP;
	getfile(xtrnull, xtrnull);
}

/*
 * skip over any extended attributes.
 */
void
skipxattr(void)
{

	while (spcl.c_flags & DR_EXTATTRIBUTES)
		skipfile();
}

static void
progressinfo(void)
{
	static time_t tschedule = 0;
	time_t tnow = time(NULL);
	if (tnow >= tschedule) {
		if (tschedule) {
			char msgbuf[128];
			snprintf(msgbuf, sizeof(msgbuf),
				"Block %lld read on volume %ld (inode %d block %lld)\n",
				tpblksread, volno, curfile.ino, blksread);
			fprintf(stderr, "  RESTORE: ");
			fwrite(msgbuf, strlen(msgbuf), 1, stderr);
			fflush(stderr);
		}
		tschedule = tnow + 300;
	}
}

/*
 * Extract a file from the tape.
 * When an allocated block is found it is passed to the fill function;
 * when an unallocated block (hole) is found, a zeroed buffer is passed
 * to the skip function.
 */
void
getfile(void (*fill) (const char *, off_t, size_t), void (*skip) (const char *, off_t, size_t))
{
	int i;
	volatile int curblk = 0;
	volatile int64_t size = spcl.c_dinode.di_size;
	off_t pos;
	volatile int last_write_was_hole = 0;
	static char clearedbuf[MAXBSIZE];
	char buf[MAXBSIZE / TP_BSIZE][TP_BSIZE];
	char junk[TP_BSIZE];

	if (spcl.c_type == TS_END)
		panic("ran off end of tape\n");
	if (spcl.c_magic != NFS_MAGIC)
		panic("not at beginning of a file\n");
	if (!gettingfile && setjmp(restart) != 0)
			return;
	gettingfile++;
	pos = 0;

loop:
	i = 0;
	while (i < spcl.c_count) {
		uint32_t enclen = readmapflag?1:get_s_addr_length(spcl.c_addr + i);
		uint32_t b;
		for (b = 0; b < enclen; ++b) {
			if (readmapflag || (spcl.c_addr[i]&1)) {
				if (curblk > 0 && (pos + curblk*TP_BSIZE) % fssize == 0) {
					size_t bytes = curblk * TP_BSIZE;
					(*fill)((char *)buf, pos, bytes);
					pos += bytes;
					curblk = 0;
					last_write_was_hole = 0;
				}
				readtape(&buf[curblk++][0]);
			} else {
				if (curblk > 0) {
					size_t bytes = curblk * TP_BSIZE;
					(*fill)((char *)buf, pos, bytes);
					pos += bytes;
					curblk = 0;
				}
				size_t sbytes = (size_t)(size > TP_BSIZE ? TP_BSIZE : size);
				(*skip)(clearedbuf, pos, sbytes);
				pos += sbytes;
				last_write_was_hole = 1;
			}
			if ((size -= TP_BSIZE) <= 0) {
				++b;
				do {
					enclen = readmapflag?1:get_s_addr_length(spcl.c_addr + i);
					if (readmapflag || (spcl.c_addr[i]&1))
						for(;b < enclen; ++b) {
							readtape(junk);
						}
					b = 0;
					i += enclen>64?2:1;
				} while (i < spcl.c_count);
				break;
			}
		}
		i += enclen>64?2:1;
	}
	progressinfo();
	while (gethead(&spcl) != GOOD) {
		fprintf(stderr, "Incorrect block for %s at %lld blocks\n",
			curfile.name, blksread);
	}
	if (size > 0) {
		if (spcl.c_type == TS_ADDR)
			goto loop;

		int missing_blocks = (size+TP_BSIZE-1)/TP_BSIZE;
		size = 0;
		fprintf(stderr, "Missing %d 1K blocks at the end of %s. File probably corrupted\n",
				missing_blocks, curfile.name);
		Dprintf(stdout,
				"Missing address (header) block for %s at %lld blocks\n",
				curfile.name, blksread);
	}
	if (curblk > 0) {
		size_t bytes = (size_t)((curblk * TP_BSIZE) + size);  // size is -ve or zero here
		(*fill)((char *)buf, pos, bytes);
		pos += bytes;
		last_write_was_hole = 0;
	}
	if (!Nflag && last_write_was_hole) {
		/*
		 * Do not attempt a truncate if running an on-the-fly compare, as there
		 * IS no file to truncate in that case!
		 */
		if (ofile >= 0 && curfile.action == USING && ftruncate(ofile, pos) < 0)
			warn("%s: ftruncate", curfile.name);
	}
	if (!readingmaps)
		findinode(&spcl);
	gettingfile = 0;
}

static size_t hashole = 0;

/*
 * Write out the next block of a file.
 */
static void
xtrfile(const char *buf, off_t pos, size_t size)
{

	if (Nflag)
		return;

	if (hashole) {
		if (lseek(ofile, pos, SEEK_SET) == -1)
			err(1, "seek error extracting inode %u, name %s\n",
					curfile.ino, curfile.name);
		hashole = 0;
	}

	if (write(ofile, buf, (int) size) == -1)
		err(1, "write error extracting inode %u, name %s\n",
			curfile.ino, curfile.name);
}

#ifdef DUMP_MACOSX
static void
xtrfilefinderinfo(const char *buf, UNUSED(off_t pos), size_t size)
{
	bcopy(buf, &gFndrInfo, size);
}
#endif /* DUMP_MACOSX */

/*
 * Skip over a hole in a file.
 */
/* ARGSUSED */
static void
xtrskip(UNUSED(const char *buf), UNUSED(off_t pos), UNUSED(size_t size))
{
	if (Nflag)
		return;

	hashole = 1;
}

/*
 * Collect the next block of a symbolic link.
 */
static void
xtrlnkfile(const char *buf, UNUSED(off_t pos), size_t size)
{
	size_t n = size;
	if (pathlen + size > MAXPATHLEN) {
		errx(1, "symbolic link name: %s->%s%.*s; too long %zu",
		    curfile.name, lnkbuf, (int)size, buf, pathlen+size);
		n = MAXPATHLEN - pathlen;
	}
	pathlen += n;
	(void) strncat(lnkbuf, buf, n);
}

/*
 * Skip over a hole in a symbolic link (should never happen).
 */
/* ARGSUSED */
static void
xtrlnkskip(UNUSED(const char *buf), UNUSED(off_t pos), UNUSED(size_t size))
{

	errx(1, "unallocated block in symbolic link %s", curfile.name);
}

/*
 * Collect the next block of a bit map.
 */
static void
xtrmap(const char *buf, UNUSED(off_t pos), size_t size)
{

	memmove(map, buf, size);
	map += size;
}

/*
 * Skip over a hole in a bit map (should never happen).
 */
/* ARGSUSED */
static void
xtrmapskip(UNUSED(const char *buf), UNUSED(off_t pos), size_t size)
{

	panic("hole in map\n");
	map += size;
}

/*
 * Noop, when an extraction function is not needed.
 */
/* ARGSUSED */
void
xtrnull(UNUSED(const char *buf), UNUSED(off_t pos), UNUSED(size_t size))
{

	return;
}

#if COMPARE_ONTHEFLY
/*
 * Compare the next block of a file.
 */

#ifdef HAVE_SEEK_HOLE
static off_t cmpfile_data = 0;  // The next data block -1 indicates that remainder of file is a hole
static off_t cmpfile_hole = 0;  // The next hole
static off_t cmpfile_curr = 0;  // The next hole
#endif

static void
xtrcmpfile(const char *buf, off_t pos, size_t size)
{
	static char cmpbuf[MAXBSIZE];

	if (cmperror)
		return;
#ifdef HAVE_SEEK_HOLE
	cmpfile_curr = pos + size;

	if (cmpfile_data > pos || cmpfile_data == -1) {
		fprintf(stderr, "%s: tape has data but disk has hole\n", curfile.name);
		cmperror = 1;
		return;
	}

	// Find the next hole if we haven't already.
	if (cmpfile_hole < (off_t)(pos + size)) {
		if ((cmpfile_hole = lseek(ifile, pos, SEEK_HOLE)) == -1 ||
				lseek(ifile, pos, SEEK_SET) == -1) {
			fprintf(stderr, "%s: size has changed.\n",
				curfile.name);
			cmperror = 1;
			return;
		}
	}
	/* ext4 uninit look like sparse files to SEEK_HOLE/SEEK_DATA */
	/*
	if (cmpfile_hole < (off_t)(pos + size)) {
		fprintf(stderr, "%s: tape has data but disk has hole\n", curfile.name);
		cmperror = 1;
		return;
	}
	*/
#else
	(void)pos;
#endif
	if (read(ifile, cmpbuf, size) != (ssize_t)size) {
		fprintf(stderr, "%s: size has changed.\n",
			curfile.name);
		cmperror = 1;
		return;
	}

	if (memcmp(buf, cmpbuf, size) != 0) {
		fprintf(stderr, "%s: tape and disk copies are different\n",
			curfile.name);
		cmperror = 1;
		return;
	}
}

/*
 * Skip over a hole in a file.
 */
static void
xtrcmpskip(UNUSED(const char *buf), off_t pos, size_t size)
{
	if (cmperror)
		return;
#ifdef HAVE_SEEK_HOLE
	cmpfile_curr = pos + size;

	if (cmpfile_hole > pos) {
		fprintf(stderr, "%s: tape has hole but disk has data\n", curfile.name);
		cmperror = 1;
		return;
	}

	// Trivial case - we're already in a hole
	if (cmpfile_data > (off_t)(pos + size) || cmpfile_data == -1)
		return;

	if ((cmpfile_data = lseek(ifile, pos, SEEK_DATA)) == -1) {
		/* There is no more data in the file */
		return;
	}

	if (cmpfile_data >= (off_t)(pos + size)) {
		return;
	}

	fprintf(stderr, "%s: tape has hole but disk has data\n", curfile.name);
	cmperror = 1;
	return;
#else
	(void)pos;
	static char cmpbuf[MAXBSIZE];
	int i;

	if (read(ifile, cmpbuf, size) != (ssize_t)size) {
		fprintf(stderr, "%s: size has changed.\n",
			curfile.name);
		cmperror = 1;
		return;
	}

	for (i = 0; i < (int)size; ++i)
		if (cmpbuf[i] != '\0') {
			fprintf(stderr, "%s: tape and disk copies are different\n",
				curfile.name);
			cmperror = 1;
			return;
		}
#endif /* HAVE_SEEK_HOLE */
}
#endif /* COMPARE_ONTHEFLY */

static void
xtrxattr(const char *buf, UNUSED(off_t pos), size_t size)
{
	if (xattrlen + size > XATTR_MAXSIZE) {
		fprintf(stderr, "EA size too big (%zd)", xattrlen + size);
		return;
	}
	memcpy(xattrbuf + xattrlen, buf, size);
	xattrlen += size;
}

#if !COMPARE_ONTHEFLY
static int
do_cmpfiles(int fd_tape, int fd_disk, off_t size)
{
	static char buf_tape[BUFSIZ];
	static char buf_disk[BUFSIZ];
	ssize_t n_tape;
	ssize_t n_disk;

	while (size > 0) {
		if ((n_tape = read(fd_tape, buf_tape, sizeof(buf_tape))) < 1) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: unexpected EOF[1]");
		}
		if ((n_disk = read(fd_disk, buf_disk, sizeof(buf_tape))) < 1) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: unexpected EOF[2]");
		}
		if (n_tape != n_disk) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: sizes different!");
		}
		if (memcmp(buf_tape, buf_disk, (size_t)n_tape) != 0) return (1);
		size -= n_tape;
	}
	return (0);
}

/* for debugging compare problems */
#undef COMPARE_FAIL_KEEP_FILE

static
#ifdef COMPARE_FAIL_KEEP_FILE
/* return true if tapefile should be unlinked after compare */
int
#else
void
#endif
cmpfiles(char *tapefile, char *diskfile, struct stat *sbuf_disk)
{
	struct stat sbuf_tape;
	int fd_tape, fd_disk;

	if (stat(tapefile, &sbuf_tape) != 0) {
		panic("can't lstat tmp file %s: %s\n", tapefile,
		      strerror(errno));
		do_compare_error;
	}

	if (sbuf_disk->st_size != sbuf_tape.st_size) {
		fprintf(stderr,
			"%s: size changed from %lld to %lld.\n",
			diskfile, (long long)sbuf_tape.st_size, (long long)sbuf_disk->st_size);
		do_compare_error;
#ifdef COMPARE_FAIL_KEEP_FILE
		return (0);
#else
		return;
#endif
	}

	if ((fd_tape = open(tapefile, O_RDONLY)) < 0) {
		panic("can't open %s: %s\n", tapefile, strerror(errno));
		do_compare_error;
	}
	if ((fd_disk = open(diskfile, O_RDONLY)) < 0) {
		close(fd_tape);
		panic("can't open %s: %s\n", diskfile, strerror(errno));
		do_compare_error;
	}

	if (do_cmpfiles(fd_tape, fd_disk, sbuf_tape.st_size)) {
		fprintf(stderr, "%s: tape and disk copies are different\n",
			diskfile);
		close(fd_tape);
		close(fd_disk);
		do_compare_error;
#ifdef COMPARE_FAIL_KEEP_FILE
		/* rename the file to live in /tmp */
		/* rename `tapefile' to /tmp/<basename of diskfile> */
		{
			char *p = strrchr(diskfile, '/');
			char newname[MAXPATHLEN];
			if (!p) {
				panic("can't find / in %s\n", diskfile);
			}
			snprintf(newname, sizeof(newname), "%s/debug/%s", tmpdir, p + 1);
			if (rename(tapefile, newname)) {
				panic("rename from %s to %s failed: %s\n",
				      tapefile, newname,
				      strerror(errno));
			} else {
				fprintf(stderr, "*** %s saved to %s\n",
					tapefile, newname);
			}
		}

		/* don't unlink the file (it's not there anymore */
		/* anyway) */
		return (0);
#else
		return;
#endif
	}
	close(fd_tape);
	close(fd_disk);
#ifdef COMPARE_FAIL_KEEP_FILE
	return (1);
#endif
}
#endif /* !COMPARE_ONTHEFLY */

/* collect both extended attribute tape blocks, then
	compare the attributes */
static void
compareattr(char *name)
{
	int xattr_count_all = 0;
	int attr_compare_err = 0;

	while (spcl.c_flags & DR_EXTATTRIBUTES) {
		switch (spcl.c_extattributes) {
		case EXT_MACOSFNDRINFO:
			msg("MacOSX not supported for comparision in this version, skipping\n");
			skipfile();
			break;
		case EXT_MACOSRESFORK:
			msg("MacOSX not supported for comparision in this version, skipping\n");
			skipfile();
			break;
		case EXT_XATTR: {
			char xattr[XATTR_MAXSIZE];
			int xattr_count_thisblock = 0;

			/* in-inode xattr record comes first, if present at all; then block xattr */
			if (readxattr(xattr) == GOOD) {
				/* this returns fail if the buffer contains
				 * different attributes than the file on disk;
				 * note that it mustn't return fail if the file
				 * on disk has more attributes! */
				if (xattr_compare(name, xattr) == FAIL)
					attr_compare_err = 1;

				xattr_count(xattr, &xattr_count_thisblock);
				xattr_count_all += xattr_count_thisblock;
			}
			else
				attr_compare_err = 1;
			break;
		}
		default:
			msg("unexpected inode extension %ld, skipping\n", spcl.c_extattributes);
			skipfile();
			break;
		}
	}

	int count_file = xattr_count_file(name);
	if (count_file < 0) {
		attr_compare_err = 1;
	} else if (count_file != xattr_count_all) {
		fprintf(stderr, "%s: file does not have the same number of attributes (%d) as the tape (%d).\n",
						name, count_file, xattr_count_all );
		attr_compare_err = 1;
	}
	if (attr_compare_err)
		do_compare_error;
}

#if !COMPARE_ONTHEFLY
static char tmpfilename[MAXPATHLEN];
#endif

void
comparefile(char *name)
{
	mode_t mode;
	uid_t uid;
	uid_t gid;
	unsigned int flags;
	unsigned long newflags;
	struct stat sb;
	int r;
#if !COMPARE_ONTHEFLY
	static char *tmpfile = NULL;
	struct stat stemp;
#endif
	curfile.name = name;
	curfile.action = USING;
	mode = curfile.dip->di_mode;
	flags = curfile.dip->di_flags;
	uid = curfile.dip->di_uid;
	gid =  curfile.dip->di_gid;

	if ((r = lstat(name, &sb)) != 0) {
		warn("unable to stat %s", name);
		do_compare_error;
		skipfile();
		return;
	}

	Vprintf(stdout, "comparing %s (size: %lld, mode: 0%o)\n", name,
		(long long)sb.st_size, mode);

	if (sb.st_mode != mode) {
		fprintf(stderr, "%s: mode changed from 0%o to 0%o.\n",
			name, mode & 07777, sb.st_mode & 07777);
		do_compare_error;
	}
	if (sb.st_uid != uid) {
		fprintf(stderr, "%s: uid changed from %d to %d.\n",
			name, uid, sb.st_uid);
		do_compare_error;
	}
	if (sb.st_gid != gid) {
		fprintf(stderr, "%s: gid changed from %d to %d.\n",
			name, gid, sb.st_gid);
		do_compare_error;
	}
	if (lgetflags(name, &newflags, flags) < 0) {
		if (flags != 0) {
			warn("%s: lgetflags failed", name);
			do_compare_error;
		}
	}
	else {
		if (newflags != flags) {
			fprintf(stderr, "%s: flags changed from 0x%08x to 0x%08lx.\n",
				name, flags, newflags);
			do_compare_error;
		}
	}
	if (spcl.c_flags & DR_METAONLY) {
		skipfile();
		return;
	}
	switch (mode & IFMT) {
	default:
		fprintf(stderr, "%s: unknown file mode 0%o\n", name, mode);
		skipfile();
		return;

	case IFSOCK:
	case IFIFO:
	case IFDIR:
		skipfile();
		compareattr(name);
		return;

	case IFLNK: {
		char lbuf[MAXPATHLEN + 1];
		int lsize;

		if (!(sb.st_mode & S_IFLNK)) {
			fprintf(stderr, "%s: is no longer a symbolic link\n",
				name);
			do_compare_error;
			return;
		}
		lnkbuf[0] = '\0';
		pathlen = 0;
		getfile(xtrlnkfile, xtrlnkskip);
		if (pathlen == 0) {
			fprintf(stderr,
				"%s: zero length symbolic link (ignored)\n",
				name);
			do_compare_error;
			return;
		}
		if ((lsize = readlink(name, lbuf, MAXPATHLEN)) < 0) {
			panic("readlink of %s failed: %s\n", name,
			      strerror(errno));
			do_compare_error;
		}
		lbuf[lsize] = 0;
		if (strcmp(lbuf, lnkbuf) != 0) {
			fprintf(stderr,
				"%s: symbolic link changed from %s to %s.\n",
				name, lnkbuf, lbuf);
			do_compare_error;
			return;
		}
		compareattr(name);
		return;
	}

	case IFCHR:
	case IFBLK:
		if (!(sb.st_mode & (S_IFCHR|S_IFBLK))) {
			fprintf(stderr, "%s: no longer a special file\n",
				name);
			do_compare_error;
			skipfile();
			return;
		}

		if (sb.st_rdev != (dev_t)curfile.dip->di_rdev) {
			fprintf(stderr,
				"%s: device changed from %d,%d to %d,%d.\n",
				name,
				major(curfile.dip->di_rdev),
				minor(curfile.dip->di_rdev),
				major(sb.st_rdev),
				minor(sb.st_rdev));
			do_compare_error;
		}
		skipfile();
		compareattr(name);
		return;

	case IFREG:
#if COMPARE_ONTHEFLY
		if ((ifile = open(name, O_RDONLY)) < 0) {
			warn("can't open %s", name);
			skipfile();
			do_compare_error;
		}
		else {
			cmperror = 0;
			/*
			 * tell getfile to not truncate as there is no open file:
			 * xtrcmpfile works off mem buffer.
			 */
			ofile = -1;
			if (curfile.dip->di_flags & EXT4_INLINE_DATA_FL) {
				uint64_t inline_filesize = curfile.dip->di_size>sizeof(inline_data)?sizeof(inline_data):curfile.dip->di_size;
				memcpy(inline_data, curfile.dip->di_db, INLINE_DATA_MAX_INODE_SIZE);
				getfile(xtrcmpfile, xtrnull);
				compareattr(name);
				xtrcmpfile(inline_data, 0, inline_filesize);
			} else {
#ifdef HAVE_SEEK_HOLE
				cmpfile_data = 0;
				cmpfile_hole = 0;
				cmpfile_curr = 0;
#endif
				getfile(xtrcmpfile, xtrcmpskip);
				compareattr(name);
			}
			if (!cmperror) {
				char c;
#ifdef HAVE_SEEK_HOLE
				if (lseek(ifile, cmpfile_curr, SEEK_SET) == -1) {
					fprintf(stderr, "%s: size has changed.\n",
						name);
					cmperror = 1;
				}
				else
#endif
				if (read(ifile, &c, 1) != 0) {
					fprintf(stderr, "%s: size has changed.\n",
						name);
					cmperror = 1;
				}
			}
			if (cmperror)
				do_compare_error;
			close(ifile);
		}
#else /* COMPARE_ONTHEFLY */
		if (tmpfile == NULL) {
			/* argument to mktemp() must not be in RO space: */
			snprintf(tmpfilename, sizeof(tmpfilename), "%s/restoreCXXXXXX", tmpdir);
			tmpfile = mktemp(&tmpfilename[0]);
		}
		if ((stat(tmpfile, &stemp) == 0) && (unlink(tmpfile) != 0)) {
			panic("cannot delete tmp file %s: %s\n",
			      tmpfile, strerror(errno));
		}
		if ((ofile = open(tmpfile, O_WRONLY | O_CREAT | O_TRUNC, 0600)) < 0) {
			panic("cannot create file temp file %s: %s\n",
			      name, strerror(errno));
		}
		if (curfile.dip->di_flags & EXT4_INLINE_DATA_FL) {
			uint64_t inline_filesize = curfile.dip->di_size>sizeof(inline_data)?sizeof(inline_data):curfile.dip->di_size;
			memcpy(inline_data, curfile.dip->di_db, INLINE_DATA_MAX_INODE_SIZE);
			getfile(xtrcmpfile, xtrnull);
			compareattr(name);
			xtrfile(inline_data, inline_filesize);
		} else {
			getfile(xtrfile, xtrskip);
			compareattr(name);
		}
		(void) close(ofile);
#ifdef COMPARE_FAIL_KEEP_FILE
		if (cmpfiles(tmpfile, name, &sb))
			unlink(tmpfile);
#else
		cmpfiles(tmpfile, name, &sb);
		unlink(tmpfile);
#endif
#endif /* COMPARE_ONTHEFLY */
		return;
	}
	/* NOTREACHED */
}

/*
 * Read TP_BSIZE blocks from the input.
 * Handle read errors, and end of media.
 * Decompress compressed blocks.
 */
/*
 * Read compressed data from a tape and uncompress it.
 * Handle read errors, and end of media.
 * Since a tape consists of separate physical blocks, we try
 * to recover from errors by repositioning the tape to the next
 * block.
 */
static void
readtape(char *buf)
{
	long size, i;
	size_t rl;
	int ret, newvol;
	struct tapebuf *tpb;

	if (blkcnt < numtrec) {
		memmove(buf, &tbufptr[(blkcnt++ * TP_BSIZE)], TP_BSIZE);
		blksread++;
		tpblksread++;
		return;
	}
	/* need to read the next block */
	tbufptr = tapebuf;
	for (i = 0; i < ntrec; i++)
		((struct s_spcl *)&tapebuf[i * TP_BSIZE])->c_magic = 0;
	numtrec = ntrec;
	tpb = (struct tapebuf *) tapebuf;
#ifdef USE_QFA
	if (createtapeposflag)
		(void)getpos_tape(&curtapepos);
#endif

	/* read the block */
	size = bufsize + PREFIXSIZE;
	ret = readblock_tape(tapebuf, size, &rl);
	if (ret <= 0)
		goto readerr;

	tbufptr = decompress_tapebuf(tpb, rl);
	if (tbufptr == NULL) {
		msg_read_error("Tape decompression error while");
		if (qflag || !reply("continue", yflag))
			exit(1);
		memset(tapebuf, 0, PREFIXSIZE + bufsize);
		tbufptr = tapebuf;
	}
	goto moverecord;

readerr:
	/* Handle errors: EOT switches to the next volume, other errors
	 * attempt to position the tape to the next block.
	 */
	if (ret == 0) {
		Vprintf(stdout, "End-of-tape encountered\n");
		newvol = volno + 1;
		volno = 0;
		numtrec = 0;
		getvol(newvol);
		readtape(buf);
		return;
	}

	msg_read_error("Tape read error while");
	if (qflag || !reply("continue", yflag))
		exit(1);
	memset(tapebuf, 0, PREFIXSIZE + bufsize);
	tbufptr = tapebuf;

	if (ioctl_tape(MTFSR, 1) < 0) {
		warn("continuation failed");
		if (qflag || !reply("assume end-of-tape and continue", yflag))
			exit(1);
		ret = 0;         /* end of tape */
		goto readerr;
	}

moverecord:
	blkcnt = 0;
	memmove(buf, &tbufptr[(blkcnt++ * TP_BSIZE)], TP_BSIZE);
	blksread++;
	tpblksread++;
}

/*
 *  Decompress a struct tapebuf into a buffer. readsize is the size read
 *  from the tape/file and is used for error messages. Returns a pointer
 *  to the location of the uncompressed buffer or NULL on errors.
 *  Adjust numtrec and complain for a short block.
 */
static char *
decompress_tapebuf(struct tapebuf *tpbin, int readsize)
{
	/* readblock_tape ensures that all blocks have a prefix even if they
	 * don't on tape. */
	int cresult, blocklen;
	unsigned long worklen;
	void *output = NULL;
	char *reason = NULL, *lengtherr = NULL;

	/* build a length error message */
	blocklen = tapebuf_getlength(tpbin);
	if (readsize < blocklen + (int)PREFIXSIZE)
		lengtherr = "short";
	else
		if (readsize > blocklen + (int)PREFIXSIZE)
			lengtherr = "long";

#ifdef HAVE_COMPRESSION
	worklen = comprlen;
#endif
	cresult = 1;
	if (tapebuf_getcompressed(tpbin)) {
		/* uncompress whatever we read, if it fails, complain later */
		switch (tapebuf_getflags(tpbin)) {
			case COMPRESS_ZLIB: {
#ifndef HAVE_ZLIB
				errx(1,"This restore version doesn't support zlib decompression");
#else
				cresult = uncompress(comprbuf, &worklen,
						(const unsigned char*)tpbin->buf, blocklen);
				output = comprbuf;
				switch (cresult) {
					case Z_OK:
						break;
					case Z_MEM_ERROR:
						reason = "not enough memory";
						break;
					case Z_BUF_ERROR:
						reason = "buffer too small";
						break;
					case Z_DATA_ERROR:
						reason = "data error";
						break;
					default:
						reason = "unknown";
				}
				if (cresult == Z_OK)
					cresult = 1;
				else
					cresult = 0;
#endif /* HAVE_ZLIB */
				break;
			}
			case COMPRESS_BZLIB: {
#ifndef HAVE_BZLIB
				errx(1,"This restore version doesn't support bzlib decompression");
#else
				unsigned int worklen2 = worklen;
				cresult = BZ2_bzBuffToBuffDecompress(
						comprbuf, &worklen2,
						tpbin->buf, blocklen, 0, 0);
				worklen = worklen2;
				output = comprbuf;
				switch (cresult) {
					case BZ_OK:
						break;
					case BZ_MEM_ERROR:
						reason = "not enough memory";
						break;
					case BZ_OUTBUFF_FULL:
						reason = "buffer too small";
						break;
					case BZ_DATA_ERROR:
					case BZ_DATA_ERROR_MAGIC:
					case BZ_UNEXPECTED_EOF:
						reason = "data error";
						break;
					default:
						reason = "unknown";
				}
				if (cresult == BZ_OK)
					cresult = 1;
				else
					cresult = 0;
#endif /* HAVE_BZLIB */
				break;
			}
			case COMPRESS_LZO: {
#ifndef HAVE_LZO
				errx(1,"This restore version doesn't support lzo decompression");
#else
				lzo_uint worklen2 = worklen;
				cresult = lzo1x_decompress((const unsigned char*)tpbin->buf, blocklen, comprbuf,
						&worklen2, NULL);
				worklen = worklen2;
				output = comprbuf;
				switch (cresult) {
					case LZO_E_OK:
						break;
					case LZO_E_ERROR:
					case LZO_E_EOF_NOT_FOUND:
						reason = "data error";
						break;
					default:
						reason = "unknown";
				}
				if (cresult == LZO_E_OK)
					cresult = 1;
				else
					cresult = 0;
				break;
			}
#endif /* HAVE_LZO */
			default:
				errx(1, "This restore version doesn't recognise %d compression", tapebuf_getflags(tpbin));
				break;
		}
	} else {
		output = tpbin->buf;
		worklen = tapebuf_getlength(tpbin);;
	}
	if (cresult) {
		numtrec = worklen / TP_BSIZE;
		if (worklen % TP_BSIZE != 0)
			reason = "length mismatch";
	}
	if (reason) {
		if (lengtherr)
			fprintf(stderr, "%s compressed block: %d expected: %zu\n",
				lengtherr, readsize, tapebuf_getlength(tpbin) + PREFIXSIZE);
		fprintf(stderr, "decompression error, block %lld: %s\n",
			tpblksread+1, reason);
		if (!cresult)
			output = NULL;
	}
	return output;
}

/*
 * Print an error message for a read error.
 * This was exteracted from the original readtape().
 */
static void
msg_read_error(char *m)
{
	switch (curfile.action) {
		default:
			fprintf(stderr, "%s trying to set up tape\n", m);
			break;
		case UNKNOWN:
			fprintf(stderr, "%s trying to resynchronize\n", m);
			break;
		case USING:
			fprintf(stderr, "%s restoring %s\n", m, curfile.name);
			break;
		case SKIP:
			fprintf(stderr, "%s skipping over inode %u\n", m,
				curfile.ino);
			break;
	}
}

/*
 * Read the first block and get the blocksize from it. Test
 * for a compressed dump tape/file. setup() will make the final
 * determination by checking the compressed flag if gethead()
 * finds a valid header.
 * Note that the first block of each tape/file is not compressed
 * and does not have a prefix.
 */
static void
findtapeblksize(void)
{
	long i;
	size_t readlen;
	size_t len;
	struct tapebuf *tpb = (struct tapebuf *) tapebuf;
	struct s_spcl spclpt;

	for (i = 0; i < ntrec; i++)
		((struct s_spcl *)&tapebuf[i * TP_BSIZE])->c_magic = 0;
	blkcnt = 0;
	tbufptr = tapebuf;
	/* read the first block. if it's bigger than this then the user needs
	 * to specify a different -b on the command line.*/
	len = ntrec * TP_BSIZE + PREFIXSIZE;

	if (readblock_tape(tapebuf, len, &readlen) <= 0)
		errx(1, "Tape read error on first record");

	memcpy(&spclpt, tapebuf, TP_BSIZE);
	cvtflag = 0;
	if (converthead(&spclpt) == FAIL) {
		cvtflag++;
		if (converthead(&spclpt) == FAIL) {
			/* Special case for old compressed tapes with prefix */
			if (readlen % TP_BSIZE != 0)
				goto oldformat;
			errx(1, "Tape is not a dump tape");
		}
		fprintf(stderr, "Converting to new file system format.\n");
	}
	/*
	 * If the input is a tape, we tried to read ntrec * TP_BSIZE bytes.
	 * If the value of ntrec is too large, we read less than
	 * what we asked for; adjust the value of ntrec and test for
	 * a compressed dump tape.
	 */
	if (readlen % TP_BSIZE != 0) {
oldformat:
		/* may be old format compressed dump tape with a prefix */
		memcpy(&spclpt, tpb->buf, TP_BSIZE);
		cvtflag = 0;
		if (converthead(&spclpt) == FAIL) {
			cvtflag++;
			if (converthead(&spclpt) == FAIL)
				errx(1, "Tape is not a dump tape");
			fprintf(stderr, "Converting to new file system format.\n");
		}
		if (readlen % TP_BSIZE == PREFIXSIZE
		    && tapebuf_getcompressed(tpb) == 0
		    && spclpt.c_type == TS_TAPE
		    && spclpt.c_flags & DR_COMPRESSED) {
			fprintf(stderr, "Dump tape is compressed.\n");
			tbufptr = tpb->buf;
			if (tapebuf_getlength(tpb) > bufsize)
				errx(1, "Tape blocksize is too large, use "
					"\'-b %d\' ", tapebuf_getlength(tpb) / TP_BSIZE);
		}
		else
			errx(1, "Tape block size (%zu) is not a multiple of dump block size (%d)",
				readlen, TP_BSIZE);
	}
	ntrec = readlen / TP_BSIZE;
	if (spclpt.c_type == TS_TAPE) {
		if (spclpt.c_flags & DR_COMPRESSED) {
			fprintf(stderr, "Dump tape is compressed.\n");
		}
		if (spclpt.c_ntrec > ntrec)
			errx(1, "Tape blocksize is too large, use "
				"\'-b %d\' ", spclpt.c_ntrec);
	}
	numtrec = ntrec;
#ifdef HAVE_COMPRESSION
	newcomprbuf(ntrec);
#endif
	Vprintf(stdout, "Tape block size is %d\n", ntrec);
}

static void
setmagtapein(void) {
	static int done = 0;
	if (done)
		return;
	done = 1;
	int magtapein = 0;
	if (!pipein)
		/* need to know if input is really from a tape */
		magtapein = magtapein_tape();

	Vprintf(stdout,"Input is from a %s %s\n",
			host ? "remote" : "local",
			magtapein ? "tape" :
			Vflag ? "multi-volume (no tape)" : "file/pipe");
}

/*
 * Read the next block from the tape.
 * Check to see if it is one of several vintage headers.
 * If it is an old style header, convert it to a new style header.
 * If it is not any valid header, return an error.
 */
static int
gethead(struct s_spcl *buf)
{
	readtape((char *)buf);
	return converthead(buf);
}

int
converthead(struct s_spcl *buf)
{
	int32_t i;
	union u_ospcl {
		char dummy[TP_BSIZE];
		struct	s_ospcl {
			int32_t	c_type;
			int32_t	c_date;
			int32_t	c_ddate;
			int32_t	c_volume;
			u_int32_t	c_tapea_lo;
			u_int16_t c_inumber;
			int32_t	c_magic;
			int32_t	c_checksum;
			struct odinode {
				u_int16_t odi_mode;
				u_int16_t odi_nlink;
				u_int16_t odi_uid;
				u_int16_t odi_gid;
				int32_t	odi_size;
				int32_t	odi_rdev;
				char	odi_addr[36];
				int32_t	odi_atime;
				int32_t	odi_mtime;
				int32_t	odi_ctime;
			} c_dinode;
			int32_t	c_count;
			char	c_fill[256];
		} s_ospcl;
	} u_ospcl;

	if (!cvtflag) {
		if (buf->c_magic != NFS_MAGIC) {
			if (swabi(buf->c_magic) != NFS_MAGIC)
				return (FAIL);
			if (!Bcvt) {
				Vprintf(stdout, "Note: Doing Byte swapping\n");
				Bcvt = 1;
			}
		}
		if (checksum((int *)buf) == FAIL)
			return (FAIL);
		if (Bcvt)
			/*
			 * We are byteswapping a u_spcl.s_spcl here from dumprestore.h
				8 <<	swap 8 terms
				i <<	c_type
					c_date
					c_ddate
					c_volume
					c_tapea_lo
					c_inumber
					c_magic
					c_checksum
				4 <<	swap 4 terms
				s	di_mode		(new_bsd_inode)
					di_nlink
					oldids[2]
				1 <<    swap 1 term
				l	di_size
				29 <<	swap 29 terms
				i	di_atime	0
					di_mtime	2
					di_ctime	4
					di_db[NDADDR]	6
					di_ib[NIADDR]	18
					di_flags	21
					di_blocks	22
					di_gen		23
					di_uid		24
					di_gid		25
					di_spare[2]	26	(end of new_bsd_inode)
					c_count		29
				528 <<	swap 528 terms
				b	c_data	0
					c_label	512
				192 <<	swap 192 terms
				b <<    c_filesys	0
					c_dev		64
					c_host		128
				6 <<	swap 6 terms
				i <<	c_flags
					c_firstrec
					c_ntrec
					c_extattributes
					c_tapea_hi
					c_firstrec_hi
				c_spare[28] not swapped.
			*/
			swabst((u_char *)"8i4s1l29i528bi192b6i", (u_char *)buf);
		goto good;
	}
	memcpy(&u_ospcl.s_ospcl, buf, TP_BSIZE);
	if (checksum((int *)(&u_ospcl.s_ospcl)) == FAIL)
		return(FAIL);
	if (u_ospcl.s_ospcl.c_magic == OFS_MAGIC) {
		memset((char *)buf, 0, TP_BSIZE);
		buf->c_type = u_ospcl.s_ospcl.c_type;
		buf->c_date = u_ospcl.s_ospcl.c_date;
		buf->c_ddate = u_ospcl.s_ospcl.c_ddate;
		buf->c_volume = u_ospcl.s_ospcl.c_volume;
		buf->c_tapea_lo = u_ospcl.s_ospcl.c_tapea_lo;
		buf->c_tapea_hi = 0;
		buf->c_inumber = u_ospcl.s_ospcl.c_inumber;
		buf->c_checksum = u_ospcl.s_ospcl.c_checksum;
		buf->c_magic = u_ospcl.s_ospcl.c_magic;
		buf->c_dinode.di_mode = u_ospcl.s_ospcl.c_dinode.odi_mode;
		buf->c_dinode.di_nlink = u_ospcl.s_ospcl.c_dinode.odi_nlink;
		buf->c_dinode.di_uid = u_ospcl.s_ospcl.c_dinode.odi_uid;
		buf->c_dinode.di_gid = u_ospcl.s_ospcl.c_dinode.odi_gid;
		buf->c_dinode.di_size = u_ospcl.s_ospcl.c_dinode.odi_size;
		buf->c_dinode.di_rdev = u_ospcl.s_ospcl.c_dinode.odi_rdev;
		buf->c_dinode.di_atime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_atime;
		buf->c_dinode.di_mtime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_mtime;
		buf->c_dinode.di_ctime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_ctime;
		buf->c_count = u_ospcl.s_ospcl.c_count;
		memmove(buf->c_addr, u_ospcl.s_ospcl.c_fill, 256);
	}
	else if (u_ospcl.s_ospcl.c_magic == FS_UFS2_MAGIC) {
		int64_t i64;
		memcpy(&i64, &u_ospcl.dummy[896], sizeof(i64));
		buf->c_date = i64;
		memcpy(&i64, &u_ospcl.dummy[904], sizeof(i64));
		buf->c_ddate = i64;
		memcpy(&i64, &u_ospcl.dummy[912], sizeof(i64));
		buf->c_tapea_lo = i64;
		buf->c_tapea_hi = 0;	// Did we have 64 bits here really?
		memcpy(&i64, &u_ospcl.dummy[920], sizeof(i64));
		buf->c_firstrec_lo = i64;
		buf->c_firstrec_hi = 0;
		buf->c_ntrec = 0;
		buf->c_extattributes = 0;
		buf->c_flags |= DR_NEWINODEFMT;
		ufs2flag = 1;
	}
	else
		return(FAIL);
	buf->c_magic = NFS_MAGIC;

good:
	readmapflag = 0;

	switch (buf->c_type) {

	case TS_CLRI:
	case TS_BITS:
		/*
		 * Have to patch up missing information in bit map headers
		 */
		buf->c_inumber = 0;
		buf->c_dinode.di_size = buf->c_count * TP_BSIZE;
		if (buf->c_count > TP_NINDIR)
			readmapflag = 1;
		else
			for (i = 0; i < buf->c_count; i++)
				buf->c_addr[i]++;
		break;

	case TS_TAPE:
		if ((buf->c_flags & DR_NEWINODEFMT) == 0)
			oldinofmt = 1;
		/* fall through */
	case TS_END:
		buf->c_inumber = 0;
		if (buf->c_flags & DR_INODEINFO) {
			memcpy(volinfo, buf->c_inos, TP_NINOS * sizeof(dump_ino_t));
			if (Bcvt)
				swabst((u_char *)"128i", (u_char *)volinfo);
		}
		break;

	case TS_INODE:
	case TS_ADDR:
		break;

	default:
		panic("gethead: unknown inode type %d\n", buf->c_type);
		break;
	}
	/*
	 * If we are restoring a filesystem with old format inodes,
	 * copy the uid/gid to the new location.
	 */
	if (oldinofmt) {
		buf->c_dinode.di_uid = buf->c_dinode.di_ouid;
		buf->c_dinode.di_gid = buf->c_dinode.di_ogid;
	}
	if (dflag)
		accthdr(buf);
	switch (buf->c_type) {
	case TS_CLRI:
	case TS_BITS:
	case TS_INODE:
		blksread = 0;
		break;
	}
	return(GOOD);
}

static uint32_t countdatablocks(struct s_spcl *header) {
	int i = 0;
	uint32_t recs = 0;
	while(i < header->c_count) {
		uint32_t enclen = readmapflag?1:get_s_addr_length(header->c_addr + i);
		if (readmapflag || (header->c_addr[i]&1))
			recs += enclen;
		i+=enclen>64?2:1;
	}
	return recs;
}
/*
 * Check that a header is where it belongs and predict the next header
 */
static void
accthdr(struct s_spcl *header)
{
	static dump_ino_t previno = 0x7fffffff;
	static int prevtype;
	static long long predict;

	if (header->c_type == TS_TAPE) {
		fprintf(stderr, "Volume header (%s inode format) ",
		    oldinofmt ? "old" : "new");
		if (u_spcl_c_firstrec(header))
			fprintf(stderr, "begins with record %lld",
				u_spcl_c_firstrec(header));
		fprintf(stderr, "\n");
		previno = 0x7fffffff;
		return;
	}
	if (previno == 0x7fffffff)
		goto newcalc;
	char msg[128] = {0};
	switch (prevtype) {
	case TS_BITS:
		snprintf(msg, sizeof msg, "Dumped inodes map header");
		break;
	case TS_CLRI:
		snprintf(msg, sizeof msg, "Used inodes map header");
		break;
	case TS_INODE:
		snprintf(msg, sizeof msg, "File header, ino %u", previno);
		break;
	case TS_ADDR:
		//snprintf(msg, sizeof msg, "File continuation header, ino %u", previno);
		break;
	case TS_END:
		snprintf(msg, sizeof msg, "End of tape header");
		break;
	}
	if (predict != blksread - 1)
		snprintf(msg + strlen(msg), sizeof msg - strlen(msg), "; predicted %lld blocks, got %lld blocks",
			predict, blksread - 1);
	if(strlen(msg))
		fprintf(stderr, "%s\n", msg);
newcalc:
	switch(header->c_type) {
	case TS_BITS:
	case TS_CLRI:
	case TS_INODE:
		predict = countdatablocks(header);
		break;
	case TS_ADDR:
		// Need to account for this TS_ADDR block too
		predict += countdatablocks(header) + 1;
		break;
	case TS_END:
		break;
	}
	prevtype = header->c_type;
	previno = header->c_inumber;
}

/*
 * Find an inode header.
 * Complain if had to skip, and complain is set.
 */
static void
findinode(struct s_spcl *header)
{
	static long skipcnt = 0;
	long i;
	char buf[TP_BSIZE];

	curfile.name = "<name unknown>";
	curfile.action = UNKNOWN;
	curfile.dip = NULL;
	curfile.ino = 0;
	do {
		if (header->c_magic != NFS_MAGIC) {
			skipcnt++;
			while (gethead(header) == FAIL ||
			    header->c_date != dumpdate)
				skipcnt++;
		}
		switch (header->c_type) {

		case TS_ADDR:
			/*
			 * Skip up to the beginning of the next record
			 */
			for (i = 0; i < header->c_count; i++)
				if (header->c_addr[i])
					readtape(buf);
			while (gethead(header) == FAIL ||
			    header->c_date != dumpdate)
				skipcnt++;
			break;

		case TS_INODE:
			curfile.dip = &header->c_dinode;
			curfile.ino = header->c_inumber;
			break;

		case TS_END:
			curfile.ino = maxino;
			break;

		case TS_CLRI:
			curfile.name = "<file removal list>";
			break;

		case TS_BITS:
			curfile.name = "<file dump list>";
			break;

		case TS_TAPE:
			panic("unexpected tape header\n");
			exit(1);

		default:
			panic("unknown tape header type %d\n", spcl.c_type);
			exit(1);

		}
	} while (header->c_type == TS_ADDR);
	if (skipcnt > 0)
#ifdef USE_QFA
		if (!noresyncmesg)
#endif
			fprintf(stderr, "resync restore, skipped %ld blocks\n",
				skipcnt);
	skipcnt = 0;
}

static int
checksum(int *buf)
{
	int i, j;

	j = sizeof(union u_spcl) / sizeof(int);
	i = 0;
	if(!Bcvt) {
		do
			i += *buf++;
		while (--j);
	} else {
		/* What happens if we want to read restore tapes
			for a 16bit int machine??? */
		do
			i += swabi(*buf++);
		while (--j);
	}

	if (i != CHECKSUM) {
		fprintf(stderr, "Checksum error %o, inode %u file %s\n", i,
			curfile.ino, curfile.name);
		return(FAIL);
	}
	return(GOOD);
}

#ifdef RRESTORE
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void
#ifdef __STDC__
msg(const char *fmt, ...)
#else
msg(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	va_list ap;
#ifdef __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	(void)vfprintf(stderr, fmt, ap);
	va_end(ap);
}
#endif /* RRESTORE */

static u_char *
swab16(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[1]; sp[1] = c;
		sp += 2;
	}
	return (sp);
}

static u_char *
swab32(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[3]; sp[3] = c;
		c = sp[1]; sp[1] = sp[2]; sp[2] = c;
		sp += 4;
	}
	return (sp);
}

static u_char *
swab64(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[7]; sp[7] = c;
		c = sp[1]; sp[1] = sp[6]; sp[6] = c;
		c = sp[2]; sp[2] = sp[5]; sp[5] = c;
		c = sp[3]; sp[3] = sp[4]; sp[4] = c;
		sp += 8;
	}
	return (sp);
}

void
swabst(u_char *cp, u_char *sp)
{
	int n = 0;

	while (*cp) {
		switch (*cp) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			n = (n * 10) + (*cp++ - '0');
			continue;

		case 's': case 'w': case 'h':
			if (n == 0)
				n = 1;
			sp = swab16(sp, n);
			break;

		case 'i':
			if (n == 0)
				n = 1;
			sp = swab32(sp, n);
			break;

		case 'l':
			if (n == 0)
				n = 1;
			sp = swab64(sp, n);
			break;

		default: /* Any other character, like 'b' counts as byte. */
			if (n == 0)
				n = 1;
			sp += n;
			break;
		}
		cp++;
		n = 0;
	}
}

static u_int
swabi(u_int x)
{
	swabst((u_char *)"i", (u_char *)&x);
	return (x);
}

#if 0
static u_long
swabl(u_long x)
{
	swabst((u_char *)"l", (u_char *)&x);
	return (x);
}
#endif

void
RequestVol(long tnum)
{
	FLUSHTAPEBUF();
	getvol(tnum);
}

#ifdef USE_QFA

/*
 * read next data from tape to re-sync
 */
void
ReReadInodeFromTape(dump_ino_t theino)
{
	long	cntloop = 0;

	FLUSHTAPEBUF();
	noresyncmesg = 1;
	do {
		cntloop++;
		gethead(&spcl);
	} while (!(spcl.c_inumber == theino && spcl.c_type == TS_INODE && spcl.c_date == dumpdate));

	tpblksread = u_spcl_c_tapea() + spcl.c_volume;
#ifdef DEBUG_QFA
	fprintf(stderr, "DEBUG: %ld reads\n", cntloop);
	fprintf(stderr, "DEBUG: bufsize %d\n", bufsize);
	fprintf(stderr, "DEBUG: ntrec %d\n", ntrec);
	fprintf(stderr, "DEBUG: tapea %lld\n", u_spcl_c_tapea());
	fprintf(stderr, "DEBUG: tpblksread %lld\n", tpblksread);
#endif
	findinode(&spcl);
	noresyncmesg = 0;
}

#endif /* USE_QFA */
