/* This file is part of the hkl3d library.
 *
 * The hkl library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The hkl library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the hkl library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (C) 2010-2019, 2025 Synchrotron SOLEIL
 *                         L'Orme des Merisiers Saint-Aubin
 *                         BP 48 91192 GIF-sur-YVETTE CEDEX
 *
 * Authors: Picca Frédéric-Emmanuel <picca@synchrotron-soleil.fr>
 *          Oussama Sboui <sboui@synchrotron-soleil.fr>
 */
#include <string.h>

#include "hkl3d-private.h"
#include "tap/basic.h"
#include "tap/hkl-tap.h"

static void check_model_validity(Hkl3D *hkl3d)
{
	int res;
	Hkl3DAxis **axis;

	res = TRUE;

	/* imported 1 config files with 7 Hkl3DObjects */
	res &= DIAG(darray_size(hkl3d->config->models) == 1);
	Hkl3DModel *model = darray_item(hkl3d->config->models, 0);
	res &= DIAG(darray_size(model->objects) == 7);


	/* each moving Hkl3DAxis must have a size of 1 */
	darray_foreach(axis, hkl3d->geometry->axes){
		if (NULL != (*axis)->mparameter)
			res &= DIAG(darray_size((*axis)->objects) == 1);
	}

	ok(res == TRUE, "no identical objects");
}

/* check the collision and that the right axes are colliding */
static void check_collision(Hkl3D *hkl3d)
{
	char buffer[1000];
	int res = TRUE;
	double values[] = {23, 0., 0., 0., 0., 0.};
	Hkl3DAxis **axis;
	Hkl3DObject **object;

	/* check the collision and that the right axes are colliding */
	res &= DIAG(hkl_geometry_axis_values_set(hkl3d->geometry->geometry,
						 values, ARRAY_SIZE(values),
						 HKL_UNIT_USER, NULL));

	res &= DIAG(hkl3d_is_colliding(hkl3d) == TRUE);
	strcpy(buffer, "");

	/* now check that only delta and mu are colliding */
	darray_foreach(axis, hkl3d->geometry->axes){
		darray_foreach(object, (*axis)->objects){

			const char *name = (*axis)->node->mName.data;
			int is_colliding = (*object)->is_colliding == TRUE;

			if(is_colliding){
				strcat(buffer, " ");
				strcat(buffer, name);
			}

			if(!strcmp(name, "mu") || !strcmp(name, "delta")){
				res &= DIAG(is_colliding == TRUE);
			} else {
				res &= DIAG(is_colliding == FALSE);
			}
		}
	}
	ok(res == TRUE,  "collision [%s]", buffer);
}

static void check_no_collision(Hkl3D *hkl3d)
{
	int res = TRUE;
	double i;

	/* check that rotating around komega/kappa/kphi do not create collisison */
	res &= DIAG(hkl_geometry_set_values_v(hkl3d->geometry->geometry,
					      HKL_UNIT_USER, NULL,
					      0., 0., 0., 0., 0., 0.));
	/* komega */
	for(i=0; i<=360; i=i+10){
		double values[] = {0., i, 0., 0., 0., 0.};

		res &= DIAG(hkl_geometry_axis_values_set(hkl3d->geometry->geometry,
							 values, ARRAY_SIZE(values),
							 HKL_UNIT_USER, NULL));
		res &= DIAG(hkl3d_is_colliding(hkl3d) == FALSE);
	}

	/* kappa */
	for(i=0; i<=360; i=i+10){
		double values[] = {0., 0., i, 0., 0., 0.};

		res &= DIAG(hkl_geometry_axis_values_set(hkl3d->geometry->geometry,
							 values, ARRAY_SIZE(values),
							 HKL_UNIT_USER, NULL));
		res &= DIAG(hkl3d_is_colliding(hkl3d) == FALSE);
	}

	/* kphi */
	for(i=0; i<=360; i=i+10){
		double values[] = {0., 0., 0., i, 0., 0.};

		res &= DIAG(hkl_geometry_axis_values_set(hkl3d->geometry->geometry,
							 values, ARRAY_SIZE(values),
							 HKL_UNIT_USER, NULL));
		res &= DIAG(hkl3d_is_colliding(hkl3d) == FALSE);
	}
	ok(res == TRUE, "no-collision");
}

int main(void)
{
	const char *model = "../data/" SOLEIL_DIFFABS_MODEL;
	char* filename;
	const HklFactory *factory;
	HklGeometry *geometry;
	Hkl3D *hkl3d;

	factory = hkl_factory_get_by_name("K6C", NULL);
	geometry = hkl_factory_create_new_geometry(factory);

	/* compute the filename of the diffractometer config file */
	filename  = test_file_path(model);
	if (NULL == filename)
		filename = strdup (model);

	hkl3d = hkl3d_new(filename, geometry);

	plan(3);
	check_model_validity(hkl3d);
	check_collision(hkl3d);
	check_no_collision(hkl3d);
	/* TODO add/remove object*/

	hkl3d_free(hkl3d);
	test_file_path_free(filename);
	hkl_geometry_free(geometry);

	return 0;
}
