#!/usr/bin/python3
# SPDX-License-Identifier: Apache-2.0
# SPDX-FileCopyrightText: Zygmunt Krynicki

from __future__ import annotations

import argparse
import configparser
import dataclasses
import enum
import os
import sys
from typing import Iterator


class Boot(enum.StrEnum):
    Platform = enum.auto()
    EFI = enum.auto()
    BIOS = enum.auto()


class Architecture(enum.StrEnum):
    X86_64 = enum.auto()
    Aarch64 = enum.auto()
    Riscv64 = enum.auto()
    S390X = enum.auto()
    PPC64 = enum.auto()


@dataclasses.dataclass
class Group:
    name: str
    variant: str

    def __str__(self) -> str:
        return self.name

    @property
    def full(self) -> str:
        if self.variant != "":
            return self.name + "-" + self.variant
        return self.name

    @property
    def name_upper(self) -> str:
        return self.name.upper()


@dataclasses.dataclass
class Checksums:
    """Collection of URLs to checksum files."""

    SHA256SUMS: str  # An URL really
    SHA512SUMS: str  # The direct checksum
    SHA256SUM: str  # Literal value

    @property
    def empty(self) -> bool:
        return self.SHA256SUMS == "" and self.SHA256SUM == "" and self.SHA512SUMS == ""


@dataclasses.dataclass
class Image:
    group: Group
    name: str  # canonical name, including architecture "$GROUP-$VARIANT-$RELEASE.$ARCH"
    url: str
    url_program: str  # program for distributions without a stable URL
    checksums: Checksums
    arch: Architecture
    boot: Boot
    release: str
    hostname: str

    @property
    def filename(self) -> str:
        """filename is the name of the file pointed to by the URL."""
        return os.path.basename(self.url)


def load_images(filename: str) -> Iterator[Image]:
    p = configparser.ConfigParser()
    p.read(filename)
    g = Group(name=p["META"]["Group"], variant=p["META"]["variant"])
    for sect in p.sections():
        if sect == "META":
            continue
        rel = p[sect].get("Release", "")
        hostname = g.name
        if rel:
            hostname += "-" + rel
        url = p[sect].get("URL", "")
        arch = Architecture(p[sect]["Arch"])
        if arch not in sect:
            raise ValueError(
                "Suspicious image {!r} for architecture {}".format(sect, arch)
            )
        if url != "":
            if arch == Architecture.X86_64:
                okay = "x86_64" in url or "amd64" in url
            elif arch == Architecture.Aarch64:
                okay = "aarch64" in url or "arm64" in url
            else:
                okay = str(arch) in url
            if not okay:
                raise ValueError(
                    "Suspicious URL {!r} for architecture {}".format(url, arch)
                )

        checksums = Checksums(
            SHA256SUMS=p[sect].get("SHA256SUMS", ""),
            SHA256SUM=p[sect].get("SHA256SUM", ""),
            SHA512SUMS=p[sect].get("SHA512SUMS", ""),
        )

        if checksums.empty:
            print("No checksums provided for image {!r}".format(sect), file=sys.stderr)

        yield Image(
            group=g,
            name=sect,
            url=url,
            url_program=p[sect].get("URLProgram", ""),
            checksums=checksums,
            arch=arch,
            boot=Boot(p[sect].get("Boot", "platform")),
            release=p[sect].get("Release", ""),
            hostname=p[sect].get("HostName", hostname),
        )


ruleset_group = """
$(GARDEN_DL_DIR)/{group}: | $(GARDEN_DL_DIR)
+mkdir -p $@
"""

ruleset_common = """
GARDEN_SYSTEMS += {img.name}

ifeq ($(GARDEN_ARCH),{img.arch})
.PHONY: all
all:: {img.name}.qcow2 {img.name}.run
endif

{img.name}.meta-data: export META_DATA = $(call CLOUD_INIT_META_DATA_TEMPLATE,{img.group})
{img.name}.meta-data: $(MAKEFILE_LIST)
+echo "$${{META_DATA}}" | tee $@
+touch --reference=$(shell stat $^ -c '%Y %n' | sort -nr | cut -d ' ' -f 2 | head -n 1) $@

{img.name}.user-data: export USER_DATA = $(call $(call PICK_CLOUD_INIT_USER_DATA_TEMPLATE_FOR,{img.group.name_upper},{img.release}),{img.hostname},{img.group})
{img.name}.user-data: $(MAKEFILE_LIST) $(wildcard $(GARDEN_PROJECT_DIR)/.image-garden.mk)
+echo "$${{USER_DATA}}" | tee $@
+touch --reference=$(shell stat $^ -c '%Y %n' | sort -nr | cut -d ' ' -f 2 | head -n 1) $@

.PHONY: clean
clean::
+rm -f $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2
+rm -f {img.name}.qcow2 {img.name}.run
+rm -f {img.name}.meta-data {img.name}.user-data
"""

ruleset_fetch_common = """
ifeq ($(GARDEN_ARCH),{img.arch})
.PHONY: fetch
fetch: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2
endif

.PHONY: distclean
distclean::
+if [ -L $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 ]; then rm -f $(GARDEN_DL_DIR)/{img.group}/$$(readlink $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2); fi
+rm -f $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2
+rm -f $(GARDEN_DL_DIR)/{img.group}/{img.name}.url

ifneq (,{img.checksums.SHA512SUMS})
$(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.url
+$(strip $(GARDEN_ITSELF) download-with-sha512sums $$(cat $<) {img.checksums.SHA512SUMS} $(GARDEN_DL_DIR)/{img.group})
+$(strip ln -sf "$$(basename "$$(cat $<)")" $@)
else
ifneq (,{img.checksums.SHA256SUM})
$(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.url
+$(strip $(GARDEN_ITSELF) download-with-sha256sum $$(cat $<) {img.checksums.SHA256SUM} $(GARDEN_DL_DIR)/{img.group})
+$(strip ln -sf "$$(basename "$$(cat $<)")" $@)
else
ifneq (,{img.checksums.SHA256SUMS})
$(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.url
+$(strip $(GARDEN_ITSELF) download-with-sha256sums $$(cat $<) {img.checksums.SHA256SUMS} $(GARDEN_DL_DIR)/{img.group})
+$(strip ln -sf "$$(basename "$$(cat $<)")" $@)
else
$(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.url
+$(strip $(GARDEN_ITSELF) download $$(cat $<) $(GARDEN_DL_DIR)/{img.group})
+$(strip ln -sf "$$(basename "$$(cat $<)")" $@)
endif
endif
endif
"""

ruleset_fetch_direct = """
$(GARDEN_DL_DIR)/{img.group}/{img.name}.url: $(MAKEFILE_LIST) | $(GARDEN_DL_DIR)/{img.group}
+echo {img.url} >$@
"""

ruleset_fetch_prog = """
$(GARDEN_DL_DIR)/{img.group}/{img.name}.url: $(MAKEFILE_LIST) | $(GARDEN_DL_DIR)/{img.group}
+{img.url_program} >$@
"""

ruleset_x86_64_efi = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2 {img.name}.efi-code.img {img.name}.efi-vars.img
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_X86_64_EFI_CMDLINE,$(word 1,$|),$(word 2,$|),$(word 3,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso {img.name}.efi-code.img {img.name}.efi-vars.img
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_X86_64_EFI_CMDLINE,$@,$(word 3,$^),$(word 4,$^)) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_x86_64_bios = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_X86_64_BIOS_CMDLINE,$(word 1,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_X86_64_BIOS_CMDLINE,$@) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_aarch64_efi = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2 {img.name}.efi-code.img {img.name}.efi-vars.img
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_AARCH64_EFI_CMDLINE,$(word 1,$|),$(word 2,$|),$(word 3,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso {img.name}.efi-code.img {img.name}.efi-vars.img
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_AARCH64_EFI_CMDLINE,$@,$(word 3,$^),$(word 4,$^)) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_riscv64_efi = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2 {img.name}.efi-code.img {img.name}.efi-vars.img
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_RISCV64_EFI_CMDLINE,$(word 1,$|),$(word 2,$|),$(word 3,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso {img.name}.efi-code.img {img.name}.efi-vars.img
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_RISCV64_EFI_CMDLINE,$@,$(word 3,$^),$(word 4,$^)) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_s390x = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_S390X_CMDLINE,$(word 1,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=/bin/bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_S390X_CMDLINE,$@) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_ppc64 = """
{img.name}.run: $(MAKEFILE_LIST) | {img.name}.qcow2
+echo "#!/bin/sh" >$@
+echo 'set -xeu' >>$@
+echo "# WARNING: The .qcow2 file refers to a file in $(GARDEN_DL_DIR)/{img.group}" >>$@
+echo '$(strip $(call QEMU_SYSTEM_PPC64_CMDLINE,$(word 1,$|)) "$$@")' >>$@
+chmod +x $@

{img.name}.qcow2: SHELL=/bin/bash
{img.name}.qcow2: $(GARDEN_DL_DIR)/{img.group}/{img.name}.qcow2 {img.name}.seed.iso
+$(strip $(QEMU_IMG) create -f qcow2 -b $< -F qcow2 $@ $(QEMU_IMG_SIZE))
+set -o pipefail && $(strip $(call QEMU_SYSTEM_PPC64_CMDLINE,$@) \\
    -drive file=$(word 2,$^),format=raw,id=drive1,if=none,readonly=true,media=cdrom \\
    -device virtio-blk,drive=drive1 \\
    | tee $@.log)
"""

ruleset_header = """
# SPDX-License-Identifier: Apache-2.0
# SPDX-FileCopyrightText: Zygmunt Krynicki
# This file was automatically generated by image-garden rulegen.py script.

# Set recipe prefix to the plus character. Rules are no longer sensitive to tabs.
.RECIPEPREFIX=+
"""

ruleset_footer = """
# Reset rules to use tabs.
.RECIPEPREFIX=
"""


def main() -> None:
    p = argparse.ArgumentParser(description="Generate make rules from image list")
    p.add_argument(
        "filenames", metavar="FILE", nargs="+", type=str, help="Image definition file"
    )
    args = p.parse_args()
    for filename in args.filenames:
        print(ruleset_header)
        for i, img in enumerate(load_images(filename)):
            if i == 0:
                print(ruleset_group.format(group=img.group))
            print(ruleset_common.format(img=img))
            if img.url:
                print(ruleset_fetch_common.format(img=img))
                print(ruleset_fetch_direct.format(img=img))
            elif img.url_program:
                print(ruleset_fetch_common.format(img=img))
                print(ruleset_fetch_prog.format(img=img))
            else:
                raise ValueError(
                    "Image {!r} defines neither URL nor URL program".format(img)
                )
            if img.arch == Architecture.X86_64:
                if img.boot == Boot.EFI:
                    print(ruleset_x86_64_efi.format(img=img))
                elif img.boot == Boot.BIOS:
                    print(ruleset_x86_64_bios.format(img=img))
            elif img.arch == Architecture.Aarch64:
                if img.boot == Boot.EFI:
                    print(ruleset_aarch64_efi.format(img=img))
            elif img.arch == Architecture.Riscv64:
                if img.boot == Boot.EFI:
                    print(ruleset_riscv64_efi.format(img=img))
            elif img.arch == Architecture.S390X:
                print(ruleset_s390x.format(img=img))
            elif img.arch == Architecture.PPC64:
                print(ruleset_ppc64.format(img=img))
        print(ruleset_footer)


if __name__ == "__main__":
    main()
