/**********************************************************************
 * $my_strobe example -- PLI application using VPI routines
 *
 * C source to print the simulation time and new value of a system 
 * task argument at the end of a simulation time step.
 *
 * Usage:   $my_strobe(<net_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int PLIbook_MyStrobe_calltf(), PLIbook_MyStrobe_compiletf(),
    PLIbook_EndOfTimeStep_callback();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_MyStrobe_register()
{
  s_vpi_systf_data tf_data;

  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$my_strobe";
  tf_data.calltf    = PLIbook_MyStrobe_calltf;
  tf_data.compiletf = PLIbook_MyStrobe_compiletf; 
  tf_data.sizetf    = NULL;
  tf_data.user_data = NULL;
  vpi_register_systf(&tf_data);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
PLIbook_MyStrobe_calltf(char *user_data)
{
  vpiHandle   systf_h, arg_itr, arg_h;
  s_vpi_time  time_s;
  s_cb_data   cb_data_s;

  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handle to system task argument; */
  /* compiletf has already verified only 1 arg with correct type */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  arg_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr);  /* free iterator--did not scan to null */

  /* setup end-of-time step callback */
  time_s.low  = 0;
  time_s.high = 0;
  time_s.type = vpiSimTime;
  cb_data_s.reason    = cbReadOnlySynch;
  cb_data_s.user_data = (char *)arg_h;
  cb_data_s.cb_rtn    = PLIbook_EndOfTimeStep_callback;
  cb_data_s.obj       = arg_h;
  cb_data_s.time      = &time_s;
  cb_data_s.value     = NULL;
  vpi_register_cb(&cb_data_s);
}
	
/**********************************************************************
 * Value change callback application
 *********************************************************************/
PLIbook_EndOfTimeStep_callback(p_cb_data cb_data_p)
{
  s_vpi_time   time_s;
  s_vpi_value  value_s;
  static char  new_value[] = "Z";
  vpiHandle    arg_h = (vpiHandle)cb_data_p->user_data;

  value_s.format = vpiBinStrVal;
  vpi_get_value(arg_h, &value_s);
  vpi_printf("$my_strobe: At %d: \t %s = %s\n",
             cb_data_p->time->low,
             vpi_get_str(vpiFullName, arg_h),
             value_s.value.str);
}

/**********************************************************************
 * compiletf application to verify valid systf args.
 *********************************************************************/
int PLIbook_MyStrobe_compiletf(char *user_data)
{
  vpiHandle systf_handle, arg_itr, arg_handle;
  int       tfarg_type;

  systf_handle = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_handle);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $my_strobe requires 1 argument.\n");
    tf_dofinish();
    return(0);
  }
  arg_handle = vpi_scan(arg_itr);
  tfarg_type = vpi_get(vpiType, arg_handle);
  if ( (tfarg_type != vpiReg) &&
       (tfarg_type != vpiIntegerVar) &&
       (tfarg_type != vpiRealVar) ) {
    vpi_printf("ERROR:$my_strobe arg must be a reg or variable.\n");
    tf_dofinish();
    return(0);
  }
  if (vpi_scan(arg_itr) != NULL) {
    vpi_printf("ERROR: $my_strobe only supports 1 argument.\n");
    vpi_free_object(arg_itr);
    tf_dofinish();
    return(0);
  }
  return(0);
}

/*********************************************************************/
